#!/var/python/bin/python
"""
Copyright 2000-2020 Citrix Systems, Inc. All rights reserved.
This software and documentation contain valuable trade secrets and
proprietary property belonging to Citrix Systems, Inc. None of this
software and documentation may be copied, duplicated or disclosed
without the express written permission of Citrix Systems, Inc.

  About:
  Azure Auto Configurator
  This scripts reads customdata injected via Azure Portal or a template
  and attempts to configure NetScaler accordingly. azureautoconfig is
  launched on startup by nsstartpe.sh right after starting waagent.

  Inputs:
  customdata file at /flash/nsconfig/.AZURE/customdata contains base64
  encoded JSON data. This script first decodes the base64 data and
  performs desired configuration as per data. data must be a valid JSON.
  Validated data is expected to have JSON format with following keys:

     {
        "username": "<Admin username>",
        "ha_config": {
            "peer_node": "<Peer node's NSIP>"
        },
        "azure_app_config": {
            "tenant_id": "<Azure Directory/Tenant ID>",
            "app_id": "<Application ID>",
            "secret_key": "<Application Secret Key>"
        },
        "vpx_config": {
            "pvt_ip_11": "<Private IP of Subnet 1/1>",
            "snip_11": "<Private IP of Subnet 1/1>",
            "pvt_ip_12": "<Private IP of Subnet 1/2>",
            "subnet_11": "<Address Prefix of Subnet 1/1>",
            "subnet_12": "<Address Prefix of Subnet 1/2>",
            "snips": [
                {
                    "ip": "IP to be configured as SNIP",
                    "subnet": "Address Prefix of IP Subnet",
                    "type": "snip type"
                },
                {
                    "ip": "IP to be configured as SNIP",
                    "subnet": "Address Prefix of IP Subnet",
                    "type": "snip type"
                },
                ...
                {
                    "ip": "IP to be configured as SNIP",
                    "subnet": "Address Prefix of IP Subnet",
                    "type": "snip type"
                }
            ],
            "ns_features": [
                "<feature1>",
                "<feature2>",
                ...
                "<featureN>"
            ]
        }
     }
"""

import base64
import os
import time
import sys
import json
from util import log_init, info, debug, error
import configurator

CUSTOM_DATA_FILE_PATH = '/flash/nsconfig/.AZURE/customdata'
LOG_FILE_PATH = '/var/log/waagent/azure-auto-config.log'


class CustomData(object):

    def __init__(self):
        self.__config = {}

    @property
    def config(self):
        if not self.__config:
            self._fetch_config()
        return self.__config

    def _fetch_config(self):
        try:
            with open(CUSTOM_DATA_FILE_PATH, 'r') as f_b:
                encoded_data = f_b.read()
            decoded_data = base64.b64decode(encoded_data)
            self.__config = json.loads(decoded_data)
        except Exception as err:
            error(
                "exception while fetching custom data config. (%s)" %
                str(err))
            return False
        return True

    def wait_for_custom_data(self, wait_duration=60 * 60):
        retry_wait_duration = 10
        start_time = time.time()
        while not os.path.isfile(CUSTOM_DATA_FILE_PATH):
            if time.time() - start_time > wait_duration:
                return False
            time.sleep(retry_wait_duration)
        return self._fetch_config()


class ConfiguratorManager(object):

    def __init__(self):
        self._config = {}
        log_init(LOG_FILE_PATH)

    def load_custom_data(self):
        c_d = CustomData()
        debug("waiting for custom data.")
        if not c_d.wait_for_custom_data():
            error("no custom data found, exiting.")
            sys.exit(1)
        debug("fetching custom data.")
        self._config = c_d.config

    def do_ha_config(self):
        if "ha_config" in self._config:
            cfgtr = configurator.HAConfigurator(self._config)
            cfgtr.configure()

    def do_azure_app_config(self):
        if "azure_app_config" in self._config:
            cfgtr = configurator.AzureAppConfigurator(self._config)
            cfgtr.configure()

    def do_vpx_config(self):
        if "vpx_config" in self._config:
            cfgtr = configurator.VpxConfigurator(self._config)
            cfgtr.configure()

    def run(self):
        info("starting azure auto configurator...")
        self.load_custom_data()

        # call new configuration here
        self.do_vpx_config()
        self.do_ha_config()
        self.do_azure_app_config()

        info("Completed Azure Auto Configuration. Exiting")
        sys.exit(0)


def main():
    C_M = ConfiguratorManager()
    C_M.run()


if __name__ == '__main__':
    main()
