from base64 import standard_b64encode, b64encode
import glob
import gzip
import os
import re
import shlex
import socket
import subprocess
import sys
import time
from datetime import datetime
from psutil import Process as PSUtilProcess

import requests
import xmltodict
import tempfile
import shutil

from ccauth.config.options import ConfigurationOptions
from ccauth.handle import CCAuthHandle
from ccauth.error import  CCAuthException
from ccauth.net import RequestInfo, RequestMethod
from ccauth.security.servicekey import SigningAlgorithm
from Crypto.Util import number
from Crypto.Util.asn1 import DerSequence
from binascii import a2b_base64

FILE_PERMISSION_CODE = 0o600
BACKUP_FOLDER = 'var/mastools/logs/backup/'
TRUST_KEY_DIR = "/var/mastools/trust/.ssh/"
CWS_TRUST_PVT_KEY = "/var/mastools/trust/.ssh/privatecws.pem"
PIDFILE="/var/run/splunk_logger.pid"
TIMESTAMP_FORMAT='TimeStamp='
AGENT_ID = ''
AGENT_CONF_PATH = "/var/mastools/conf/agent.conf"
AGENT_CONF_DICT = {}
APIGW_URL = {}

LOGGER_SLEEP_TIME_SECS = 120

#=================For setting up logging ==============================================================================================================
import logging.handlers

log_file_name_local = os.path.basename(__file__.rstrip('.py'))
LOG_FILENAME = '/var/mastools/logs/' + log_file_name_local + '.log'
LOG_MAX_BYTE = 50*1024*1024
LOG_BACKUP_COUNT = 10

# Set up a specific logger.with our desired output level
logger=logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)

# Add the log message handler to the logger
logger.handler = logging.handlers.RotatingFileHandler(LOG_FILENAME, maxBytes=LOG_MAX_BYTE, backupCount=LOG_BACKUP_COUNT)
logger.fortmater = logging.Formatter(fmt='%(asctime)s:%(funcName)s:%(lineno)d: [%(levelname)s] %(message)s', datefmt="%Y-%m-%d %H:%M:%S")
logger.handler.setFormatter(logger.fortmater)
logger.addHandler(logger.handler)

def get_logger_header():
    if 'customerid' not in AGENT_CONF_DICT.keys():
        customerid = "none"
    else:
        customerid = AGENT_CONF_DICT['customerid']
    if 'instanceid' not in AGENT_CONF_DICT.keys():
        instanceid = "none"
    else:
        instanceid = AGENT_CONF_DICT['instanceid']
        if len(instanceid) > 8:
            instanceid = instanceid[:4] + '****' + instanceid[-4:]
    logger_str = "[adc_splunk_logger_"+customerid+"_"+instanceid+"]: "
    return logger_str
#======================================================================================================================================================

def log_mem(msg):
    try:
        global MAX_RSS_KB, MAX_VSZ_KB
        mem_info = PSUtilProcess(os.getpid()).memory_info()
        rss_kb = mem_info.rss // 1024
        vsz_kb = mem_info.vms // 1024
        if rss_kb > MAX_RSS_KB:
            MAX_RSS_KB = rss_kb
        if vsz_kb > MAX_VSZ_KB:
            MAX_VSZ_KB = vsz_kb
        logger.debug(get_logger_header() + "[MEM] " + msg + ": RSS: " + str(rss_kb) + " KB, VSZ: " + str(vsz_kb) + " KB, MAX_RSS_KB: " + str(MAX_RSS_KB) + " KB, MAX_VSZ_KB: " + str(MAX_VSZ_KB) + " KB")
    except Exception as e:
        logger.error(get_logger_header() + "Error logging memory info: " + str(e))

def parse_agent_conf_file():
    '''Parse agent.conf and store values in AGENT_CONF_DICT
    '''

    global AGENT_CONF_DICT
    try:
        with open(AGENT_CONF_PATH, "r") as agent_conf_file:
            CONTENT= agent_conf_file.read()
            AGENT_CONF_DICT = xmltodict.parse(CONTENT)['mps_agent']
            return True

    except Exception as e:
        logger.error(get_logger_header()+'Agent not initialized ' + str(e))
        return False

#Get Encryption Key
def get_encryption_key():
    MPS_DB_SERVER_KEY_FILE = "/mpsconfig/db_key.conf"
    try:
        # Fetching DB key from /mpsconfig/db_key.conf file
        if os.path.exists(MPS_DB_SERVER_KEY_FILE):
            with open(MPS_DB_SERVER_KEY_FILE, "r+") as db_server_key:
                content = db_server_key.read()
                db_server_key.close()
                DB_SERVER_DICT = xmltodict.parse(content)['dbserverkey']
                DB_KEY = DB_SERVER_DICT['key']
                if DB_KEY is None:
                    logger.error(get_logger_header()+'Error in reading key from db file')
                    return ""
                else:
                    return DB_KEY
        else:
            return ""
    except Exception as e:
        logger.error(get_logger_header()+"exception in get_encryption_key : " + str(e))
        return ""

def create_servicekey(request_info, servicename, instanceid, private_key, signing_algorithm):
    try:
        options = ConfigurationOptions(
            service_name=servicename,
            service_instance=instanceid, # optional
            private_key=private_key
        )

        with CCAuthHandle(options) as handle:
            servicekey = handle.create_servicekey(request_info, signing_algorithm)
            return True, servicekey
    except CCAuthException as e:
        return False, 'Error creating service key ' + str(e)

def generate_service_token(url, servicename, instanceid):
    logger.info(get_logger_header()+"Generating service token")
    try:
        f = open(TRUST_KEY_DIR+"private.pem","r")
        private_key = f.read()
        f.close()
        #Checking if key is in XML format, if not then change it to XML format
        if private_key.find("BEGIN RSA PRIVATE KEY") >= 0:
            private_key = createCWSPrivateKey()
        request_info = RequestInfo(
            method = RequestMethod.GET,
            uri = url
        )

        return create_servicekey(request_info, servicename, instanceid, private_key, SigningAlgorithm.RSA_SHA256_V2)
    except Exception as e:
        return False, 'Failed to generate service token: ' + str(e)

def createCWSPrivateKey():
    try:
        if os.path.exists(CWS_TRUST_PVT_KEY):
            f = open(CWS_TRUST_PVT_KEY, "r")
            priKey =f.read()
            return priKey
        args = shlex.split("cat " + TRUST_KEY_DIR + "private.pem")
        process = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=None)
        pemPrivKey = process.stdout.read()
        lines = pemPrivKey.replace(" ", '').split()
        keyDer = DerSequence()
        keyDer.decode(a2b_base64(''.join(lines[1:-1])))
        private_key_xml  = '<RSAKeyValue>'
        private_key_xml += '<Modulus>'
        private_key_xml += standard_b64encode(number.long_to_bytes(keyDer[1]))
        private_key_xml += '</Modulus>'
        private_key_xml += '<Exponent>'
        private_key_xml += standard_b64encode(number.long_to_bytes(keyDer[2]))
        private_key_xml += '</Exponent>'
        private_key_xml += '<D>'
        private_key_xml += standard_b64encode(number.long_to_bytes(keyDer[3]))
        private_key_xml += '</D>'
        private_key_xml += '<P>'
        private_key_xml += standard_b64encode(number.long_to_bytes(keyDer[4]))
        private_key_xml += '</P>'
        private_key_xml += '<Q>'
        private_key_xml += standard_b64encode(number.long_to_bytes(keyDer[5]))
        private_key_xml += '</Q>'
        private_key_xml += '<DP>'
        private_key_xml += standard_b64encode(number.long_to_bytes(keyDer[6]))
        private_key_xml += '</DP>'
        private_key_xml += '<DQ>'
        private_key_xml += standard_b64encode(number.long_to_bytes(keyDer[7]))
        private_key_xml += '</DQ>'
        private_key_xml += '<InverseQ>'
        private_key_xml += standard_b64encode(number.long_to_bytes(keyDer[8]))
        private_key_xml += '</InverseQ>'
        private_key_xml += '</RSAKeyValue>'
        private_key_xml = standard_b64encode(private_key_xml)
        f = open(CWS_TRUST_PVT_KEY, "w+")
        f.write(private_key_xml)
        f.close()
        os.chmod(CWS_TRUST_PVT_KEY, FILE_PERMISSION_CODE)
        if os.path.exists(CWS_TRUST_PVT_KEY):
            f = open(CWS_TRUST_PVT_KEY, "r")
            priKey =f.read()
            return priKey
    except Exception as e:
        logger.error(get_logger_header()+'Failed to create Private key ' + str(e))

def get_decrypted_info():
    global AGENT_CONF_DICT
    global CUSTOMERID

    CUSTOMERID = AGENT_CONF_DICT.get('customerid', '')
    service_name = 'netappliance' #decrypt(AGENT_CONF_DICT['servicename'])
    instance_id = AGENT_CONF_DICT.get('instanceid', '')

    return service_name, instance_id

def get_apigw_url():
    global AGENT_CONF_DICT
    global CUSTOMERID
    try:
        if 'url' in AGENT_CONF_DICT:
            cloudurl = AGENT_CONF_DICT['url']
        else:
            logger.error(get_logger_header()+"Cloud URL not present in agent.conf")
            return ""

        if 'ads_service_type' in AGENT_CONF_DICT:
            service_type = AGENT_CONF_DICT['ads_service_type']
        else:
            service_type = 'ADM'
            logger.info(get_logger_header()+"ads_service_type is missing in agent.conf, sending logs to NetScaler Console topic by default")    

        if service_type=='ADM':
            apigw_url = "https://%s/%s/topics/admagentlogs" % (cloudurl, CUSTOMERID)
        elif service_type=='AUTOMATION':
            apigw_url = "https://%s/%s/topics/adsagentlogs_automation" % (cloudurl, CUSTOMERID)
        elif service_type=='INTENT':
            apigw_url = "https://%s/%s/topics/adsagentlogs_intent" % (cloudurl, CUSTOMERID)
        else:
            apigw_url = "https://%s/%s/topics/admagentlogs" % (cloudurl, CUSTOMERID)
            logger.info(get_logger_header()+"ads_service_type is missing in agent.conf, sending logs to NetScaler Console topic by default")

        logger.info(get_logger_header()+"Generated APIGW URL is "+apigw_url)
        return apigw_url
    except Exception as e:
        logger.error(get_logger_header()+'Failed to get APIGW_URL ' + str(e))
        return ""


def pidfile_create(pidfile):
    logger.info(get_logger_header()+' creating pid file')
    fp = open(pidfile, 'w+')
    pid = str(os.getpid())
    fp.write(pid)
    fp.flush()
    fp.close()

def pidfile_check(pidfile):
    if not os.path.exists(pidfile):
        logger.info(get_logger_header()+' pid file doesnt exist')
        return False

    with open(pidfile, 'r') as f:
        pid = int(f.read().strip())
        logger.info(get_logger_header()+'pid from file %s' % str(pid))
        try:
            # os.kill with 0 to the pid from the pidfile
            # will return an exception OSError if the pid
            # is not running.
            logger.info(get_logger_header()+'killing pid: %s' % str(pid))
            os.kill(pid, 0)
        except OSError:
            logger.error(get_logger_header()+'OSError caught, will recreate pid file')
            return False
        else:
            return True

def findAgentIP():
    try:
        s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        s.connect(("8.8.8.8", 80))
        IPAddr = s.getsockname()[0]
        s.close()
        return IPAddr
    except Exception as e:
        logger.error(get_logger_header()+'Failed to get Agent IP ' + str(e))
        return ""

def delete_old_backup_files():
    logger.info(get_logger_header()+'Deleting old log files from backup folder!') 
    now = time.time()
    backup_dir = "/%s" % BACKUP_FOLDER
    for filename in os.listdir(backup_dir):
        if os.path.getmtime(os.path.join(backup_dir, filename)) < now - 2 * 86400:
            if os.path.isfile(os.path.join(backup_dir, filename)):
                logger.info(get_logger_header()+filename)
                os.unlink(os.path.join(backup_dir, filename))

def find_between(input_str, first_char, last_char ):
    try:
        start = input_str.rindex( first_char ) + len( first_char )
        end = input_str.index( last_char, start )
        return input_str[start:end]
    except ValueError:
        return ""

def get_filename(input_str, first_char ):
    try:
        start = input_str.rindex( first_char ) + len( first_char )
        return input_str[start:]
    except ValueError:
        return "" 

def compress_this_file(filename, destination):
    compressed_filename = destination+'.gz'
    logger.debug(get_logger_header()+"Compressing this file at:%s" % compressed_filename)
    f_in = None
    f_out = None
    try:
        f_in = open(filename)
        f_out = gzip.open(compressed_filename, 'wt')
        f_out.writelines(f_in)
        f_out.close()
        f_in.close()
        logger.debug(get_logger_header()+"This file is compressed successfully filename:%s" % filename)
    except Exception as e:
        logger.error(get_logger_header()+'Error while gzip file %s %s ' % (filename, str(e)))
        if f_in:
            f_in.close()
        if f_out:
            f_out.close()

def format_file(filename):
    '''
    Formats log messages so that each line starts with
    TimeStamp=<time> and ends with mastools, tenant, process info.
    '''
    global AGENT_ID
    global CUSTOMERID
    AGENT_ID = AGENT_CONF_DICT.get('uuid', '')
    AGENT_IP = findAgentIP()
    process_name = find_between(filename, '/', '.')
    SUFFIX_STR = ' mastools={0} tenant={1} process={2}'.format(AGENT_IP, CUSTOMERID, process_name)

    with open(filename, 'r') as src:
        first_line = src.readline()

        # If already formatted, skip rewriting
        if TIMESTAMP_FORMAT in first_line:
            return filename

        # Create a temp file for output
        with tempfile.NamedTemporaryFile('w', delete=False) as tmp:
            # Format the first line we already read
            tmp.write("TimeStamp={}{})\n".format(first_line.rstrip(), SUFFIX_STR))

            # Process rest of the file line-by-line
            for line in src:
                tmp.write("TimeStamp={}{})\n".format(line.rstrip(), SUFFIX_STR))

        # Replace the original file with formatted temp file
        shutil.move(tmp.name, filename)

    return filename

def has_exceeded_max_upload_size(filename, response):
    exceeds = False
    try:
        response_dict = response.json()
        offsets = response_dict.get('offsets', [])
        if offsets:
            offset = offsets[0]
            ec = offset.get('error_code')
            em = offset.get('error', '')
            if ec and 'max.request.size' in em.lower():
                logger.error(get_logger_header() + "File {} exceeds maximum upload size. Error: {}".format(
                    filename, em
                ))
                exceeds = True
    except Exception as e:
        logger.error(get_logger_header() + "Error parsing upload response for file {}: {}".format(filename, repr(e)))
    return  exceeds

def upload_file_to_splunk(filename, apigw_url, service_token, customer_id):
    try:
        logger.info(get_logger_header() + "Size: %s KB for filename: %s" % ((os.path.getsize(filename))/1024, filename))
        CHUNK_SIZE = 512 * 1024  # 512 KB
        max_retries = 3
        filename = format_file(filename)
        with open(filename, 'rb') as file_p:
            # Read the file in chunks to avoid memory issues with large files
            while True:
                file_content = file_p.read(CHUNK_SIZE)
                # returns empty string when end of file is reached
                if not file_content:
                    break

                encoded = b64encode(file_content).decode('utf-8')
                payload = '{"records":[{"value":"%s"}]}' % encoded
                del file_content, encoded  # free memory early

                headers = {
                    'content-type': 'application/vnd.kafka.binary.v2+json',
                    'Authorization': 'CWSAuth service=' + service_token,
                    'customerId': customer_id
                }
                # delay between retries in seconds
                retry_delay = 2
                success = False
                for retry in range(max_retries):
                    try:
                        response_text = ''
                        status_code = 0
                        with requests.post(apigw_url, data=payload, headers=headers, timeout=30, stream=True) as response:
                            response_text = response.text
                            status_code = response.status_code
                        # logger.info(get_logger_header() + "response code: %s Response: %s" % (status_code,response_text))
                        if status_code == 200:
                            # Don't retry when the upload size has exceeded the maximum allowed size
                            if has_exceeded_max_upload_size(filename, response):
                                logger.error(get_logger_header() + "max upload size exceeded: response code: %s Response: %s" % (status_code,response_text))
                                status_code = 500
                            else:
                                success = True
                            break
                        elif status_code >= 500:  # Server error, worth retrying
                            logger.error(get_logger_header() + "Server error {}, retrying in {} seconds".format(status_code, retry_delay))
                        else:  # Client error or other, might not be worth retrying
                            logger.error(get_logger_header() + "Client error {}, may not retry".format(status_code))
                            if retry == max_retries - 1:  # Last attempt
                                break
                    except (requests.exceptions.ConnectionError, requests.exceptions.Timeout) as req_err:
                        logger.error(get_logger_header() + "Request failed on attempt {}: {}".format(retry + 1, repr(req_err)))
                        if retry == max_retries - 1:  # Last attempt
                            raise

                    if not success and retry < max_retries - 1:
                        time.sleep(retry_delay)
                        # Exponential backoff
                        retry_delay *= 2
                if success:
                    log_mem("Successfully uploaded %d bytes from the file: %s" % (CHUNK_SIZE, filename))
                del payload  # free memory early
            log_mem("Successfully uploaded to splunk, the file: %s" % (filename))
            logger.debug(get_logger_header() + "[MEM]: MAX_RSS_KB: " + str(MAX_RSS_KB) + " KB, MAX_VSZ_KB: " + str(MAX_VSZ_KB) + " KB")

        return status_code, response_text
    except Exception as e:
        if isinstance(e, (IOError, OSError)):
            logger.error(get_logger_header() + "I/O error with file {}: {}".format(filename, repr(e)))
        else:
            logger.error(get_logger_header() + "Error while uploading file to splunk {}: {}".format(filename, repr(e)))
        raise


def get_creation_time(path):
    stat_info = os.stat(path)
    # On FreeBSD, st_birthtime exists and is the real creation time
    if hasattr(stat_info, "st_birthtime"):
        return stat_info.st_birthtime
    else:
        # Fallback if st_birthtime not available
        logger.warning(get_logger_header() + "st_birthtime not available, using st_ctime as creation time")
        return stat_info.st_ctime

def send_files(apigw_url, service_token, customer_id):
    #Get a list of all rolled over files
    mastools_upgrade = glob.glob('/var/mastools/logs/mastools_upgrade.log.*')
    mastools_reg_file = glob.glob('/var/mastools/logs/mastools_reg.py.log.*')
    mastoolsd_file = glob.glob('/var/log/mastoolsd.log.*')
    admautoreg_file = glob.glob('/var/log/admautoreg.log.*')
    s3ipchandler_file = glob.glob('/var/mastools/logs/s3IPCHandler.py.log.*')
    mastools_ha_file = glob.glob('/var/mastools/logs/mastools_ha.py.log.*')
    mastools_rlc_file = glob.glob('/var/mastools/logs/resource_library_client.log.*')
    aauth_bootstrap_file = glob.glob('/var/mastools/logs/aauth_bootstrap.log.*')
    nstelemetry_file = glob.glob('/var/mastools/logs/mastools_tlmy.py.log.*')
    nstelemetry_forwarder_file = glob.glob('/var/mastools/logs/tlmy_forwarder.py.log.*')
    signature_updater_files = glob.glob('/var/mastools/logs/signature_auto_updater.log.*')
    mastools_core_cleanup_files = glob.glob('/var/mastools/logs/mastools_core_cleanup.py.log.*')
    mastools_scp_log_file = glob.glob('/var/mastools/logs/mastools_scp.log.*')

    # Handle exact file paths without pattern matching
    installns_state_file = []
    if os.path.exists('/var/nsinstall/installns_state'):
        installns_state_file = ['/var/nsinstall/installns_state']
    
    installns_state_post_reboot_file = []
    if os.path.exists('/var/nsinstall/installns_state_post_reboot'):
        installns_state_post_reboot_file = ['/var/nsinstall/installns_state_post_reboot']
    
    files = mastools_upgrade + mastools_reg_file + mastoolsd_file + admautoreg_file + s3ipchandler_file + mastools_ha_file + mastools_rlc_file + aauth_bootstrap_file + nstelemetry_file + nstelemetry_forwarder_file + signature_updater_files + mastools_core_cleanup_files + mastools_scp_log_file + installns_state_file + installns_state_post_reboot_file
    
    logger.info(get_logger_header()+"Found files to send: %s" % files)
    
    #Create Backup folder if not exist
    backup_dir = "/%s" % BACKUP_FOLDER
    if not os.path.exists(backup_dir):
        args = shlex.split("mkdir /"+BACKUP_FOLDER)
        process = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=None)
        output = process.communicate()
        
    logger.info(get_logger_header()+"Sending files to splunk ...")
    for filename in files:
        if filename.endswith('.gz'):
            logger.info(get_logger_header()+"Skipping gzip file: %s" % filename)
            continue

        #Send the rolled over files to kafka one by one
        logger.info(get_logger_header()+"")
        logger.info(get_logger_header()+">>>>>>>> Sending file: %s <<<<<<<<" % filename)

        # store created timestamp in a variable for this filename (if it is log.0 file) 
        file_timestamp = None
        if 'log.0' in filename:
            file_timestamp = get_creation_time(filename)
        try:
            status_code, status_message = upload_file_to_splunk(filename, apigw_url, service_token, customer_id)
            if status_code == 200:
                logger.debug(get_logger_header()+"Uploaded to splunk: %s" % filename)
                new_timestamp = get_creation_time(filename)
                # Check if the creation timestamp of this file has changed. If yes then it means that this file has been rotated since we uploaded it and now has been rotated to log.1
                # in such case we need to upload this file again to splunk and then backup/delete the previous rotated file log.1 as well as log.0
                if 'log.0' in filename and file_timestamp != new_timestamp:
                    logger.debug(get_logger_header()+"The file has been rotated before backup: " + filename)
                    logger.debug(get_logger_header()+"Sending the file again: " + filename)
                    status_code, status_message = upload_file_to_splunk(filename, apigw_url, service_token, customer_id)
                    # Backup/Delete previously rotated file, log.1
                    previous_rotated_file = filename[:filename.index('.log')+4] + '.1'
                    logger.debug(get_logger_header()+"Previously rotated filename: " + previous_rotated_file)
                    # Check if the previous rotated file exists
                    if not os.path.exists(previous_rotated_file):
                        logger.debug(get_logger_header()+"Previous rotated file does not exist, skipping backup/deletion for it: " + previous_rotated_file)
                    else:
                        destination = '/' + BACKUP_FOLDER + get_filename(previous_rotated_file, '/') + '_' + datetime.now().strftime("%m%d%Y_%H%M%S")
                        logger.debug(get_logger_header()+"Backing up file: " + previous_rotated_file + " at destination: " + destination)
                        compress_this_file(previous_rotated_file, destination)
                        logger.debug(get_logger_header()+"Backup done for file: " + previous_rotated_file)
                        os.unlink(previous_rotated_file)
                        logger.debug(get_logger_header()+"File deleted: " + previous_rotated_file)
                        if status_code != 200:
                            continue

                destination = '/' + BACKUP_FOLDER + get_filename(filename, '/') + '_' + datetime.now().strftime("%m%d%Y_%H%M%S")
                logger.debug(get_logger_header()+"Backing up file: " + filename + " at destination: " + destination)
                compress_this_file(filename, destination)
                logger.debug(get_logger_header()+"Backup done for file: " + filename)
                os.unlink(filename)
                logger.debug(get_logger_header()+"File deleted: " + filename)
        except Exception as e:
            logger.error(get_logger_header()+'Error while sending file %s %s ' % (filename, str(e)))
            continue
    logger.info(get_logger_header()+"No files to send, keep watching ....")

def force_rollover():
    logger.info(get_logger_header() + "Force rollover of mastoolsd.log")
    try:
        args = shlex.split("newsyslog -R \"forced rollover\" /var/log/mastoolsd.log")
        process = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=None)
        output = process.communicate()

        args = shlex.split("gunzip /var/log/mastoolsd.log.0.gz")
        process = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=None)
        output = process.communicate()
    except Exception as e:
        logger.error(get_logger_header()+'Error in forcing rollover' + str(e))


if __name__ == '__main__':
    #syslog.openlog(__file__, syslog.LOG_PID, syslog.LOG_LOCAL0)
    logger.info(get_logger_header()+'-------- Starting splunk_logger.py script --------')
    global CUSTOMERID
    global MAX_RSS_KB
    global MAX_VSZ_KB
    MAX_RSS_KB = 0
    MAX_VSZ_KB = 0

    try:
        #Send files to Kafka Rest Proxy
        logger.info(get_logger_header()+'Parsing agent conf file')
        parse_agent_conf_file()
        logger.info(get_logger_header()+'Getting encryption key')
        service_name , instance_id = get_decrypted_info()
        # if CUSTOMERID is blank, log an error and exit
        if CUSTOMERID == None or CUSTOMERID == '' or CUSTOMERID == 'None':
            logger.error(get_logger_header()+'CUSTOMERID is not set in agent.conf')
            sys.exit(1)
        # if service_name is blank, log an error and exit
        if service_name == None or service_name == '' or service_name == 'None':
            logger.error(get_logger_header()+'service_name is not set in agent.conf')
            sys.exit(1)
        # if instance_id is blank, log an error and exit
        if instance_id == None or instance_id == '' or instance_id == 'None':
            logger.error(get_logger_header()+'instance_id is not set in agent.conf')
            sys.exit(1)

        apigw_url = get_apigw_url()
        # if apigw_url is blank, log an error and exit
        if apigw_url == None or apigw_url == '' or apigw_url == 'None':
            logger.error(get_logger_header()+'apigw_url is not set in agent.conf')
            sys.exit(1)
        logger.info(get_logger_header()+'apigw_url:%s' % apigw_url)
        #print("apigw_url=%s, service_name=%s, instance_id=%s,  customer_id=%s" % (apigw_url, service_name, instance_id, CUSTOMERID)) # TODO: remove it
        status, service_token = generate_service_token(apigw_url, service_name, instance_id)
        if not status:
            logger.error(get_logger_header() + 'Error generating service token: ' + service_token)
            sys.exit(0)
        logger.info(get_logger_header() + 'Service token generated successfully')
        while True:
            if os.path.exists('/var/mastools/scripts/.disable_splunk_logger'):
                logger.info(get_logger_header()+'Splunk logging is disabled!') 
            else:
                logger.info(get_logger_header()+'Starting to send files to splunk')
                force_rollover()
                send_files(apigw_url, service_token, CUSTOMERID)
            delete_old_backup_files()
            logger.debug(get_logger_header()+'Now sleeping for %d secs' % LOGGER_SLEEP_TIME_SECS)
            logger.debug(get_logger_header() + "MAX_RSS_KB: " + str(MAX_RSS_KB) + " KB, MAX_VSZ_KB: " + str(MAX_VSZ_KB) + " KB")
            time.sleep(LOGGER_SLEEP_TIME_SECS)
    except Exception as e:
        logger.error(get_logger_header()+'Error in Script execution! ' + str(e))
        sys.exit(0)
