#!/usr/bin/env python

# Copyright 2004-2025 Citrix Systems, Inc. All rights reserved.
# This software and documentation contain valuable trade
# secrets and proprietary property belonging to Citrix Systems, Inc.
# None o1f this software and documentation may be copied,
# duplicated or disclosed without the express
# written permission of Citrix Systems, Inc.

import sys
import json
import time
import util
import util_internal_reg
import argparse
import datetime
import random
import os

from probe import Probe
import constant as cons
from logger import logger
from exception import ProbeError

def main(argv):
    try:
        parser = argparse.ArgumentParser(description='ADM Auto Reg Daemon')
        parser.add_argument("-loglevel", help="Log Level", action='store', choices=['debug', 'info'], default=cons.DEFAULT_DEBUG_LEVEL)
        args = parser.parse_args()
        util.set_log_level(args.loglevel)
        logger.info("Starting NetScaler Console Auto Reg Daemon")

        # Retrying every 2 secs for 1 minute as nscli comes up after sometime in BLX - NSLINUX-8192
        start_time = time.time()
        while True:
            try:
                deployment, ip, nodeid = util.get_deployment_info()
                mgmt_ip_address = util.get_mgmt_ip(nodeid)
                util_internal_reg.rm_snmp_mgr(mgmt_ip_address)
                break
            except Exception as e:
                if time.time() - start_time > 60:
                    logger.error("adm_auto_reg_daemon: Failed to remove snmp manager: {}".format(repr(e)))
                    break
                import traceback
                if 'nscli' in traceback.format_exc():
                    logger.info("nscli is not available, waiting for 2 seconds to retry...")
                time.sleep(2)

        is_internal_ns = util_internal_reg.check_if_internal()
        logger.info("Device is_internal_ns: {}".format(is_internal_ns))
        util.init_autoreg_state(is_internal_ns=is_internal_ns)
        probe = Probe()
        lastProbeTime = 0
        failedProbeCount = 0
        lastProbeSuccessStatus = True

        # Initial wait
        waitTime = util.get_initial_wait_time()
        if waitTime > 0:
            logger.info("Waiting for {} before sending first probe".format(str(datetime.timedelta(seconds=waitTime))))
            time.sleep(waitTime)

        try:
            if is_internal_ns and util_internal_reg.fetch_type_of_deployment() == cons.ONPREM_CLOUD_TYPE \
                    and util_internal_reg.get_test_lodestone_state() is False:
                if not os.path.exists("/nsconfig/.nameserverexception"):
                    util_internal_reg.check_and_add_nameserver_in_resolv_conf()
        except Exception as e:
            logger.error("adm_auto_reg_daemon: Failed to add nameserver 8.8.8.8 to resolv.conf: {}".format(repr(e)))

        logger.info("Starting probe requests")

        while True:
            currentTime = int(time.time())
            waiting_time_per_probe = cons.WAITING_TIME_PER_PROBE
            # Main Probe will be send after every WAITING_TIME_PER_PROBE
            if currentTime - lastProbeTime >= waiting_time_per_probe:
                try:
                    if failedProbeCount > cons.MAX_FAILED_PROBE_COUNT:
                        probe.refresh_safehaven_endpoint()
                    probe.send_probe()
                    util.set_safehaven_service_state(True)
                    failedProbeCount = 0
                    lastProbeSuccessStatus = True
                except ProbeError as e:
                    util.set_safehaven_service_state(False)
                    failedProbeCount += 1
                    lastProbeSuccessStatus = False
                    logger.error("Failed to send probe request: {}".format(str(e)))
                except Exception as e:
                    lastProbeSuccessStatus = False
                    logger.exception("Internal Error: {}".format(str(e)))

                lastProbeTime = currentTime

            # Light Probe will be send only if the main probe is successful
            if lastProbeSuccessStatus == True:
                try:
                    claimedInCurrProbe = probe.send_lightprobe()
                    if claimedInCurrProbe == True:
                        # Resetting the main probe timer to zero
                        # to send the main probe with updated claim details
                        lastProbeTime = 0
                except ProbeError as e:
                    logger.error("Failed to send light probe request: {}".format(str(e)))
                except Exception as e:
                    logger.exception("Internal Error: {}".format(str(e)))

            waitingTime = util.get_waiting_time(failedProbeCount)
            if failedProbeCount != 0:
                logger.info("Continues probe failed count: {}".format(failedProbeCount))
                logger.info("Waiting time before next probe: {}".format(str(datetime.timedelta(seconds=waitingTime))))
            time.sleep(waitingTime)

    except Exception as e:
        logger.exception("Failed to start Auto Registration Daemon: {}".format(str(e)))


if __name__ == "__main__":
    main(sys.argv)
