#
# Copyright 2025 Citrix Systems, Inc. All rights reserved.
# This software and documentation contain valuable trade
# secrets and proprietary property belonging to NetScaler Console, Inc.
# None of this software and documentation may be copied,
# duplicated or disclosed without the express
# written permission of NetScaler Console, Inc.
#
#!/usr/local/bin/python
import time
import psutil
import os
from datetime import datetime
from os.path import basename, exists
import warnings
from pathlib import Path
import sys

#=================For setting up logging ==============================================================================================================
import logging
import logging.handlers

log_file_name_local = os.path.basename(__file__)
LOG_FILENAME = '/var/mastools/logs/' + log_file_name_local + '.log'
LOG_MAX_BYTE = 10*1024*1024
LOG_BACKUP_COUNT = 20

# Set up a specific logger.with our desired output level
logger=logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)

# Add the log message handler to the logger
logger.handler = logging.handlers.RotatingFileHandler(LOG_FILENAME, maxBytes=LOG_MAX_BYTE, backupCount=LOG_BACKUP_COUNT)
logger.formatter = logging.Formatter(fmt='%(asctime)s [%(levelname)s] [%(processName)s] %(process)d [%(funcName)s:%(lineno)d]: %(message)s')
logger.handler.setFormatter(logger.formatter)
logger.addHandler(logger.handler)
#======================================================================================================================================================
METRICS = {}
restarted = False
now = datetime.now()
masd_starttime = time.mktime(now.timetuple())
# Suppress Warnings (Warning message shows up when we get latency)
warnings.filterwarnings("ignore")

def core_cleanup_action(action):
    global METRICS
    try:
        if (action == "cleanup"):
            logger.info("starting disk cleanup.")
            delete_files_except_latest("/var/core","mastools")
            logger.info("disk cleanup completed.")
    except Exception as e:
        logger.exception('Failed with Exception : '+ str(e) +', in healing action: ' + action )

def delete_files_except_latest(directory, prefix, keep_count=3):
    # Get all files starting with the prefix mastools
    files = [f for f in Path(directory).rglob(f"{prefix}*") if f.is_file()]
    
    # Sort files by last modification time in descending order
    files.sort(key=lambda f: f.stat().st_mtime, reverse=True)
    
    files_to_keep = files[:keep_count]
    files_to_delete = files[keep_count:]
    
    for file in files_to_delete:
        try:
            file.unlink()
        except Exception as e:
            logger.exception(f'Error deleting {file}: {e}')

def main():

    try:
        core_cleanup_action("cleanup") 
    except Exception as e:
        logger.info ('Exception in core dumps cleanup:' + repr(e))
        sys.exit(-1)


if __name__ == '__main__':
    main()
    sys.exit(0)
