#
# Copyright 2004-2025 Citrix Systems, Inc. All rights reserved.
# This software and documentation contain valuable trade
# secrets and proprietary property belonging to Citrix Systems, Inc.
# None o1f this software and documentation may be copied,
# duplicated or disclosed without the express
# written permission of Citrix Systems, Inc.

import pexpect
import argparse
import subprocess
import shlex
import os
import sys
import re
import copy
import json
import time
import base64
import xmltodict
import httplib2
import encodings
import uuid
import requests
import platform
import time

try:
    from mastools_reg import sign_request, is_python_2_6, auth_request_header, get_mastools_proxy_httplib2
    from ccauth import newkeypair
except:
    pass
from admautoreg import util
from admautoreg import constant as const_autoreg


MASTOOLS_CONF_FILE = '/var/mastools/conf/agent.conf'
MASTOOLS_TRUST_DIR = '/var/mastools/trust'
MASTOOLS_CERT_DIR = '/var/mastools/cert'
MASTOOLS_UPGRADE_SCRIPT = '/var/mastools/scripts/mastools_upgrade.py'
MASTOOLS_DAEMON_SCRIPT = '/var/mastools/scripts/mastoolsd'
MASTOOLS_FILE = '/var/mastools/mastools_file.tgz'
MASTOOLS_DEST_DIR = '/var/mastools'
MASTOOLS_FILE_NAME = 'mastools_file.tgz'
MASTOOLS_TRUST_KEY_DIR = '/var/mastools/trust/.ssh/'

CERT_BUNDLE_PATH='/var/mastools/cert/cacert.pem'

MGMT_TENANT_COOKIE = '_MGMT_TENANT'

ONE_MINUTE = 60*1

ADC_NETWORK_TIMEOUT = 0.5*ONE_MINUTE

ADC_DEVICE_TIMEOUT = 0.5*ONE_MINUTE

LEGATUS_AGENT_REQ_COOKIE = 'legatus_agent_request'

MASTOOLS_CWS_SERVICENAME = 'netappliance'

NETWORK_TEST_EP = "adm.cloud.com"
ADM_GRP_EP = "adm.cloud.com"
ADM_AGENT_EP = "agent.adm.cloud.com"
ADM_TRUST_EP = "trust.citrixnetworkapi.net"
ADM_DOWNLOAD_EP = "download.citrixnetworkapi.net"

ADM_EP_LIST = (ADM_GRP_EP, ADM_AGENT_EP, ADM_TRUST_EP, ADM_DOWNLOAD_EP)

SDX_LOGIN_URL = "%s://%s/nitro/v2/config/login"
ADC_LOGIN_URL = "%s://%s/nitro/v1/config/login"
ADC_NSIP_URL = "%s://%s/nitro/v1/config/nsip"
ADC_NS_LICENSE_URL = "%s://%s/nitro/v1/config/nslicense"

ADC_HTTP_PROTOCOL = "http"
ADC_HTTPS_PROTOCOL = "https"

ADC_SECURE_ONLY = "SECUREONLY"


SYSCTL_CMD = 'sysctl -n'
NSCLI_CMD = '/netscaler/nscli -U %%:.:.'
CLICMDNITRO_EXEC = 'clicmdnitro -execute -clicmd'

SHOW_NSIP = 'sh nsip -type NSIP'
SHOW_CLUSTER_NODE = 'sh cluster node'
SHOW_HANODE = 'sh ha node'
SHOW_NSIP_CLIP = 'sh nsip -type CLIP'
SHOW_HARDWARE = 'sh hardware'

CLUSTER_CCO = 'Cluster CCO'
CLUSTER_NONCCO = 'Cluster Non-CCO'
HA_PRIMARY = 'HA Primary'
HA_SECONDARY = 'HA Secondary'
STANDALONE = 'Standalone Primary'
PRIMARY = 'Primary'


ADC_DRY_RUN_RESULT = 'dry_run'
ADC_DRY_RUN_NSCLI = 'nscli'
ADC_DRY_RUN_DNS = 'dns'
ADC_DRY_RUN_INTENET_CONNECTION = 'internet'
ADC_DRY_RUN_ADM_ENDPOINT = 'adm'
ADC_DRY_RUN_ADM_AGENT_ENDPOINT = 'agent'
ADC_DRY_RUN_TRUST_ENDPOINT = 'trust'
ADC_DRY_RUN_DOWNLOAD_ENDPOINT = 'download'
ADC_DRY_RUN_ACCESS = 'access'
ADC_DRY_RUN_TRUST_DIRECTORY = 'trust_dir'
ADC_DRY_RUN_CUSTOMER_ID = 'cust_id'
ADC_DRY_RUN_INSTANCE_ID = 'inst_id'
ADC_DRY_RUN_UU_ID = 'uuid'


ADC_DRY_RUN_STATE_WORKING = 0
ADC_DRY_RUN_STATE_NOT_WORKING = 1
ADC_DRY_RUN_STATE_NOT_DONE = 2
ADC_DRY_RUN_STATE_IN_PROGRESS = 3

SVM_MPS_LIB = '/mps/lib'

IS_MGMT_TENANT = False

DEVICE_PROFILE_NAME = ''
SERVICE_URL = ''
ACTIVATION_CODE = ''
TRUST_URL = ''
CUSTOMER_ID = ''
INSTANCE_ID = ''
USERNAME = ''
PWD = ''
NSIP = ''
HTTP_PROTOCOL = ''
PRINT_RESULT_JSON = False
DRY_RUN_UUID = ''

OLD_STDOUT = None
REDIRECT_FILE = None



#=================For setting up logging ==============================================================================================================
import logging
import logging.handlers

log_file_name_local = os.path.basename(__file__)
LOG_FILENAME = '/var/mastools/logs/' + log_file_name_local + '.log'
LOG_MAX_BYTE = 50*1024*1024
LOG_BACKUP_COUNT = 20

# Set up a specific logger with our desired output level
logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)

# Add the log message handler to the logger
logger_handler = logging.handlers.RotatingFileHandler(LOG_FILENAME, maxBytes=LOG_MAX_BYTE, backupCount=LOG_BACKUP_COUNT)
logger_fortmater = logging.Formatter(fmt='%(asctime)s:%(funcName)s:%(lineno)d: [%(levelname)s] %(message)s', datefmt="%Y-%m-%d %H:%M:%S")
logger_handler.setFormatter(logger_fortmater)
logger.addHandler(logger_handler)

class bcolors:
    OK = '\033[92m' #GREEN
    WARNING = '\033[93m' #YELLOW
    FAIL = '\033[91m' #RED
    RESET = '\033[0m' #RESET COLOR


def is_python_3_running():
    if platform.python_version().startswith('3.'):
        return True
    return False

is_python_3 = is_python_3_running()

def print_dry_run_dictionary():
    if PRINT_RESULT_JSON:
        #print(bcolors.RESET+"result="+json.dumps(DRY_RUN_RESULT))
        DRY_RUN_RESULT[ADC_DRY_RUN_UU_ID] = DRY_RUN_UUID
        write_status_file(DRY_RUN_RESULT)
    return

def set_dry_run_result_success(dry_run_item):
    DRY_RUN_RESULT[dry_run_item] = ADC_DRY_RUN_STATE_WORKING
    return

def exit_error(dry_run_item):
    if dry_run_item:
        DRY_RUN_RESULT[dry_run_item] = ADC_DRY_RUN_STATE_NOT_WORKING
    print_dry_run_dictionary()
    test_cleanup_adc()
    print(bcolors.RESET + "End MASTools dry run")
    logger.debug("end running mastools_dry_run.py, DRY_RUN_RESULT:"+repr(DRY_RUN_RESULT))
    resume_print()
    sys.exit(-1)

def read_command_output(cmd):
    args = shlex.split(cmd)
    process = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=None)
    output = process.communicate()
    return output[0]

def run_command(cmd):
    args = shlex.split(cmd)
    FNULL = open(os.devnull, 'w')
    subprocess.call(args, stdout=FNULL, stderr=subprocess.STDOUT)
    FNULL.close()

def clinitro_exec(clicmd):
    nitroJson = None
    try:
        cmd = NSCLI_CMD + ' "' + CLICMDNITRO_EXEC + " '" + clicmd + "'" + '"'
        cliNitroOutput = subprocess.check_output(shlex.split(cmd), stderr=subprocess.STDOUT)
        if is_python_3:
            cliNitroOutput = cliNitroOutput.decode("utf-8")
        nitroJson = get_nitroresp_from_clicmdnitro_op(cliNitroOutput)
    except subprocess.CalledProcessError as e:
        # nscli returns exit-code=1 in error-scenario (ex: when resource isn't there)
        if e.returncode == 1:
            nitroJson = get_nitroresp_from_clicmdnitro_op(e.output)
            if nitroJson:
                return nitroJson
        raise Exception("Failed to execute clinitro cmd: {}".format(clicmd))
    except Exception as e:
        logger.error(str(e))
        raise Exception("Failed to execute clinitro cmd: {}".format(clicmd))
    return nitroJson

def get_nitroresp_from_clicmdnitro_op(cliNitroOutput):
    try:
        if cliNitroOutput == None:
            return None
        nitroJson = None
        cliNitroOutput = cliNitroOutput.split(" Done\n")[1]
        tokens = cliNitroOutput.split("\n")
        for val in tokens:
            match = re.search(r"^\s*NITRORESP :\s+(.*$)", val.strip())
            if match:
                nitroJson = json.loads(match.group(1))
                break
        return nitroJson
    except Exception as e:
        logger.error(e)
        raise Exception("Failed to get NITRORESP from cliNitroOutput str: {}".format(cliNitroOutput))

def get_sdx_ip():
    global NSIP
    global HTTP_PROTOCOL
    HTTP_PROTOCOL = ADC_HTTP_PROTOCOL
    ip_addr = subprocess.check_output(['/bin/sh', '/var/mastools/scripts/mastools_sdx_ip.sh'])
    NSIP = ip_addr.strip()
    if is_python_3:
        NSIP = NSIP.decode()
    return

def get_mgmt_ip():
    global NSIP
    global HTTP_PROTOCOL
    logger.debug("In get_mgmt_ip")
    nsip = ""
    HTTP_PROTOCOL = ADC_HTTP_PROTOCOL

    try:
        nitroJson = clinitro_exec(SHOW_NSIP)
        if nitroJson != None and 'nsip' in nitroJson:
            nsips =  nitroJson['nsip']
            if len(nsips) == 1:
                nsip = nsips[0]['ipaddress']
                if nsips[0]['gui'] == ADC_SECURE_ONLY:
                    HTTP_PROTOCOL = ADC_HTTPS_PROTOCOL
            #only in cluster we can have multiple IPs
            elif (len(nsips) > 1):
                nitroJsonCluster = clinitro_exec(SHOW_CLUSTER_NODE)
                if nitroJsonCluster != None and 'clusternode' in nitroJsonCluster:
                    clusternodes = nitroJsonCluster['clusternode']
                    for node in clusternodes:
                        if node['islocalnode'] == True:
                            nsip = node["ipaddress"]
                        if node.get('gui') == ADC_SECURE_ONLY:
                            HTTP_PROTOCOL = ADC_HTTPS_PROTOCOL
    except Exception as e:
        logger.exception(repr(e))
    NSIP = nsip
    return

def check_user_access():
    global is_device_sdx
    if not is_device_sdx:
        login_url = ADC_LOGIN_URL %(HTTP_PROTOCOL, NSIP)
    else:
        login_url = SDX_LOGIN_URL %(HTTP_PROTOCOL, NSIP)
    payload_login = {}
    payload_login["username"] = USERNAME
    payload_login["password"] = PWD
    payload = {}
    payload["login"] = payload_login
    status, content = send_request_device(login_url, "POST", json.dumps(payload))
    if status:
        try:
            content_json = json.loads(content)
            if not is_device_sdx:
                if content_json["sessionid"]:
                    logger.debug("user has the right privilege to access the ADC")
                    print(bcolors.OK+"ADC access success.")
                    set_dry_run_result_success(ADC_DRY_RUN_ACCESS)
                    return
            else:
                if content_json["login"][0]["sessionid"]:
                    logger.debug("user has the right privilege to access the SDX")
                    print(bcolors.OK+"SDX access success.")
                    set_dry_run_result_success(ADC_DRY_RUN_ACCESS)
                    return
        except Exception as e:
            logger.debug("check_user_privilege exception"+repr(e))

    logger.debug("User doesn't have the right privilege, please make sure user is super user")
    print(bcolors.FAIL+"Failed to log in to ADC using given device profile.")
    print(bcolors.FAIL+"Please check if you have entered the correct login credential in the given device profile")

    exit_error(ADC_DRY_RUN_ACCESS)



def send_request_device(url, method, payload='', basic_auth=False, time_out=ADC_DEVICE_TIMEOUT):
    if HTTP_PROTOCOL == ADC_HTTPS_PROTOCOL:
        httpConnection=httplib2.Http(".cache", timeout=time_out, disable_ssl_certificate_validation=True)
    else:
        httpConnection=httplib2.Http(".cache", timeout=time_out)
    headers_conn = {'Content-Type':'application/json'}
    logger.debug("url:"+url + ";method: " + method)
    content = ''
    try:
        if basic_auth:
            httpConnection.add_credentials(USERNAME, PWD)
        if payload == '':
            responseStatus, content = httpConnection.request(url, method, headers=headers_conn)
        else:
            responseStatus, content = httpConnection.request(url, method, headers=headers_conn, body=payload)
        if is_python_3:
            content = content.decode()
        resp_str = 'responseStatus='+str(responseStatus)+' content='+content
        if "status" in responseStatus:
            if responseStatus["status"] == '200' or responseStatus["status"] == '204' or responseStatus["status"] == '201':
                return True, content
        logger.debug(resp_str)
        logger.debug(content)
        return False, content
    except Exception as e:
        logger.debug(repr(e))
        return False, content



def send_request(url, method, payload='', time_out=ADC_NETWORK_TIMEOUT):
    global INSTANCE_ID
    service_token = sign_request(url, MASTOOLS_CWS_SERVICENAME, INSTANCE_ID)
    proxy_httplib2 = get_mastools_proxy_httplib2()
    if proxy_httplib2:
        httpConnection=httplib2.Http(".cache",proxy_info=proxy_httplib2,ca_certs=CERT_BUNDLE_PATH, timeout=time_out)
    else:
        httpConnection=httplib2.Http(".cache",ca_certs=CERT_BUNDLE_PATH, timeout=time_out)
    headers_conn = {'Content-Type':'application/json'}
    headers_conn['Authorization'] = auth_request_header+service_token
    logger.debug(url + ";method: " + method)
    legatus_cookie = LEGATUS_AGENT_REQ_COOKIE+'=true'
    content = ''
    if IS_MGMT_TENANT:
        headers_conn['Cookie'] = legatus_cookie+';'+MGMT_TENANT_COOKIE + ':"yes"'
    else:
        headers_conn['Cookie'] = legatus_cookie
    try:
        if payload == '':
            responseStatus, content = httpConnection.request(url, method, headers=headers_conn)
        else:
            responseStatus, content = httpConnection.request(url, method, headers=headers_conn, body=payload)
        if is_python_3:
            content = content.decode()
        resp_str = 'responseStatus='+str(responseStatus)+' content='+content
        logger.debug(resp_str)
        #logger.debug(content);
        if "status" in responseStatus:
            if responseStatus["status"] == '200' or responseStatus["status"] == '204' or responseStatus["status"] == '201':
                return True, content
        logger.debug(resp_str)
        logger.debug(content)
        return False, content
    except Exception as e:
        logger.debug(repr(e))
        return False, content

def gen_clour_req_url(uri):
    cloud_uri = '/' + CUSTOMER_ID + '/' + MASTOOLS_CWS_SERVICENAME + uri;
    clour_req_url = "https://"+ SERVICE_URL + cloud_uri
    return clour_req_url

def get_user_info():
    global USERNAME
    global PWD
    MAX_TRY = 3
    current_try = 0
    device_cred = None
    while (current_try < MAX_TRY):
        device_cred = get_device_cred()
        if not device_cred:
            current_try += 1
            time.sleep(10)
        else:
            break

    if not device_cred:
        exit_error(ADC_DRY_RUN_ACCESS)

    USERNAME = device_cred[MASTOOLS_DEVICE_PROFILE_DETAILS_USER_NAME]
    PWD = device_cred[MASTOOLS_DEVICE_PROFILE_DETAILS_PASSWORD]
    return

def get_user_info(username="", pwd=""):
    global USERNAME
    global PWD
    USERNAME = username
    PASSWORD = pwd
    return

def test_adc_access(username="", pwd=""):
    global is_device_sdx
    try:
        get_user_info()
    except Exception as e:
        logger.debug("Exception occured in get_user_info, exception %s", repr(e))
        print(bcolors.FAIL+"Failed to get login credential from the given device profile")
        exit_error(ADC_DRY_RUN_ACCESS)

    if (not USERNAME) or (not PWD):
        logger.debug("could not get username or pwd")
        print(bcolors.FAIL+"Failed to get login credential from the given device profile")
        exit_error(ADC_DRY_RUN_ACCESS)

    #print("USERNAME="+USERNAME+", PWD="+PWD)

    is_device_sdx = False
    if os.path.exists(SVM_MPS_LIB):
        is_device_sdx = True

    if not is_device_sdx:
        get_mgmt_ip()
    else:
        get_sdx_ip()

    if (not NSIP) or (not HTTP_PROTOCOL):
        logger.debug("could not get NSIP or HTTP_PROTOCOL")
        print(bcolors.FAIL+"Failed to log in to ADC using given device profile.")
        print(bcolors.FAIL+"Please check if you have entered the correct login credential in the given device profile")
        exit_error(ADC_DRY_RUN_ACCESS)
    check_user_access()
    return

def init_dry_run_result_more(diag_run_result):
    global DRY_RUN_RESULT
    DRY_RUN_RESULT = diag_run_result
    DRY_RUN_RESULT[ADC_DRY_RUN_ACCESS] = ADC_DRY_RUN_STATE_NOT_DONE
    DRY_RUN_RESULT[ADC_DRY_RUN_TRUST_DIRECTORY] = ADC_DRY_RUN_STATE_NOT_DONE
    DRY_RUN_RESULT[ADC_DRY_RUN_CUSTOMER_ID] = ADC_DRY_RUN_STATE_NOT_DONE
    DRY_RUN_RESULT[ADC_DRY_RUN_INSTANCE_ID] = ADC_DRY_RUN_STATE_NOT_DONE
    return

def get_trust_url():
    http_conn_fetch=httplib2.Http(".cache", ca_certs=CERT_BUNDLE_PATH)
    fetch_url = "https://" + SERVICE_URL + "/fetch_urls";
    try:
        resp_preauth, content_preauth =  http_conn_fetch.request(fetch_url, "GET")
        content_preauth = content_preauth.decode()
        if (resp_preauth['status'] == '200'):
            trust_url = content_preauth.split(';')[1]
            trust_url = trust_url.rstrip()
            trust_url = trust_url.strip('\"')
            logger.debug("Trust URL : %s, got result sucessfull", trust_url)
            return trust_url
    except Exception as e:
        logger.debug("Exception occured while post request url %s, exception %s", fetch_url, repr(e))
    return ""

def gen_trust_dir():
    run_command("rm -rf /var/mastools/trust/.ssh/")
    run_command("mkdir -p /var/mastools/trust/.ssh/")
    (public_key, private_key) = newkeypair()
    f_private = open("/var/mastools/trust/.ssh/private.pem", "w")
    f_private.write(private_key)
    f_private.close()
    f_public = open("/var/mastools/trust/.ssh/public.pem", "w")
    f_public.write(public_key)
    f_public.close()
    time.sleep(5)
    logger.debug("gen_trust_dir successfully")
    return

def test_gen_trust_dir():
    print(bcolors.RESET+"Testing trust directory generation")
    try:
        global TRUST_URL
        TRUST_URL = get_trust_url()
        if not TRUST_URL:
            print(bcolors.FAIL+"Failure in testing trust directory generation.")
            print(bcolors.FAIL+"Could not get trust endpoint from ADM. Please check your ADC connectivity to trust url")
            exit_error(ADC_DRY_RUN_TRUST_DIRECTORY)
        gen_trust_dir()
        set_dry_run_result_success(ADC_DRY_RUN_TRUST_DIRECTORY)
        print(bcolors.OK+"Testing trust directory generation Done. Success")
    except Exception as e:
        logger.debug("Exception occured while test_gen_trust_dir, exception %s", repr(e))
        print(bcolors.FAIL+"Failure in testing trust directory generation.")
        print(bcolors.FAIL+"Could not generate trust directory on ADC. Please check your ADC file system")
        exit_error(ADC_DRY_RUN_TRUST_DIRECTORY)
    return

def get_public_key_xml():
    f = open("/var/mastools/trust/.ssh/public.pem","r")
    public_key_xml = f.read()
    f.close()
    return public_key_xml

def get_activation_code(preauth_token):
    global ACTIVATION_CODE
    if (';' in preauth_token):
        preauth_service = preauth_token.split(";")
        servicename = preauth_service[0]
        preauth_token = preauth_service[1]
    ACTIVATION_CODE = preauth_token
    return



def get_customer_instance_id():
    global INSTANCE_ID
    global CUSTOMER_ID
    global TRUST_URL
    global ACTIVATION_CODE
    singingkey = get_public_key_xml()
    payload_preauth_json = {}
    payload_preauth_json["signingkey"] = singingkey
    payload_preauth_json["preauthtoken"] = ACTIVATION_CODE
    payload_preauth = json.dumps(payload_preauth_json)

    headers_preauth = {'Content-Type':'application/json'}
    url_preauth = "https://" + TRUST_URL + "/root/trust/v1/identity";
    try:
        http_conn_preauth=httplib2.Http(".cache",ca_certs=CERT_BUNDLE_PATH)
        resp_preauth, content_preauth =  http_conn_preauth.request(url_preauth, "POST", headers=headers_preauth, body=payload_preauth)
        if (resp_preauth['status'] == '200'):
            logger.debug("URL: %s, got result successfully", url_preauth)
            json_data_preauth = json.loads(content_preauth)
            logger.debug("return = %s", str(json_data_preauth))
            if (json_data_preauth['status'] == "success"):
                INSTANCE_ID = json_data_preauth['instanceid']
                CUSTOMER_ID = json_data_preauth['customerid']
            else:
                logger.error("invalid preauth token")
    except Exception as e:
        logger.debug("Exception occured while post request url %s, payload %s, exception %s", url_preauth, payload_preauth, repr(e))
    return

def test_get_customer_instance_id():
    global CUSTOMER_ID
    global INSTANCE_ID
    print(bcolors.RESET+"checking to get customer id and instance id")
    get_customer_instance_id()
    if CUSTOMER_ID and INSTANCE_ID:
        set_dry_run_result_success(ADC_DRY_RUN_CUSTOMER_ID)
        set_dry_run_result_success(ADC_DRY_RUN_INSTANCE_ID)
        print(bcolors.OK+"Successfully get customer id and instance id")
    else:
        if not CUSTOMER_ID:
            print(bcolors.FAIL+"Could not get customer id")
            print(bcolors.FAIL+"Please make sure you provide the correct activation code. Activation code could not be reused")
            exit_error(ADC_DRY_RUN_CUSTOMER_ID)
        if not INSTANCE_ID:
            print(bcolors.FAIL+"Could not get instance id")
            print(bcolors.FAIL+"Please make sure you provide the correct activation code. Activation code could not be reused")
            exit_error(ADC_DRY_RUN_INSTANCE_ID)
    return

def test_cleanup_adc():
    run_command("rm -rf /var/mastools/trust")
    return

def check_diag_run_result(diag_run_result):
    diag_result_values = diag_run_result.values()
    logger.debug("diag_result_values:"+repr(diag_result_values))
    if ADC_DRY_RUN_STATE_NOT_WORKING in diag_result_values:
        exit_error("")
    return

def write_status_file(result):
    if PRINT_RESULT_JSON:
        dry_run_status_file = open(const_autoreg.DRY_RUN_STATUS_FILE,"w+")
        dry_run_status_file.write(json.dumps(result))
        dry_run_status_file.close()
    return

def init_dry_run_status_file():
    status_init = {}
    status_init_detail = {}
    status_init_detail[ADC_DRY_RUN_NSCLI] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init_detail[ADC_DRY_RUN_DNS] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init_detail[ADC_DRY_RUN_INTENET_CONNECTION] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init_detail[ADC_DRY_RUN_ADM_ENDPOINT] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init_detail[ADC_DRY_RUN_ADM_AGENT_ENDPOINT] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init_detail[ADC_DRY_RUN_TRUST_ENDPOINT] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init_detail[ADC_DRY_RUN_DOWNLOAD_ENDPOINT] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init_detail[ADC_DRY_RUN_ACCESS] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init_detail[ADC_DRY_RUN_TRUST_DIRECTORY] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init_detail[ADC_DRY_RUN_CUSTOMER_ID] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init_detail[ADC_DRY_RUN_CUSTOMER_ID] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init[ADC_DRY_RUN_RESULT] = status_init_detail
    write_status_file(status_init)
    return



def _main():
    redirect_print()
    print(bcolors.RESET+"Start mastools dry run")
    global DRY_RUN_RESULT
    diag_run_result = util.get_diag_details(True)
    logger.debug("diag_run_result="+repr(diag_run_result))
    init_dry_run_result_more(diag_run_result)
    check_diag_run_result(diag_run_result)
    test_gen_trust_dir()
    test_get_customer_instance_id()
    time.sleep(10)
    #test_adc_access()
    print(bcolors.RESET+"End mastools dry run")
    print_dry_run_dictionary()
    logger.debug("end running mastools_dry_run.py, DRY_RUN_RESULT:"+repr(DRY_RUN_RESULT))
    resume_print()
    return

def write_status_file(result):
    if PRINT_RESULT_JSON:
        dry_run_status_file = open(const_autoreg.DRY_RUN_STATUS_FILE,"w+")
        dry_run_status_file.write(json.dumps(result))
        dry_run_status_file.close()
    return

def init_dry_run_status_file():
    status_init_detail = {}
    status_init_detail[ADC_DRY_RUN_NSCLI] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init_detail[ADC_DRY_RUN_DNS] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init_detail[ADC_DRY_RUN_INTENET_CONNECTION] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init_detail[ADC_DRY_RUN_ADM_ENDPOINT] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init_detail[ADC_DRY_RUN_ADM_AGENT_ENDPOINT] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init_detail[ADC_DRY_RUN_TRUST_ENDPOINT] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init_detail[ADC_DRY_RUN_DOWNLOAD_ENDPOINT] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init_detail[ADC_DRY_RUN_ACCESS] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init_detail[ADC_DRY_RUN_TRUST_DIRECTORY] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init_detail[ADC_DRY_RUN_CUSTOMER_ID] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init_detail[ADC_DRY_RUN_CUSTOMER_ID] = ADC_DRY_RUN_STATE_IN_PROGRESS
    status_init_detail[ADC_DRY_RUN_UU_ID] = DRY_RUN_UUID
    #write_status_file(status_init_detail)
    return

def redirect_print():
    if PRINT_RESULT_JSON:
        sys.stdout = REDIRECT_FILE
        return

def resume_print():
    if PRINT_RESULT_JSON:
        sys.stdout = OLD_STDOUT
        REDIRECT_FILE.close()
    return

if __name__ == "__main__":
    logger.debug("begin running mastools_dry_run.py")
    argc_count = len(sys.argv)
    if ((argc_count == 6) or (argc_count == 4)):
        DEVICE_PROFILE_NAME = sys.argv[1]
        SERVICE_URL = sys.argv[2]
        get_activation_code(sys.argv[3])
        if ((argc_count == 6) and (sys.argv[5] == "print")):
            PRINT_RESULT_JSON = True
            DRY_RUN_UUID = sys.argv[4]
            OLD_STDOUT = sys.stdout
            REDIRECT_FILE = open("/var/mastools/logs/mastools_dry_run_redirect.log","a")
        sys.exit(_main())
    print(bcolors.RESET+"Invalid argv")
    sys.exit(-1)