import argparse
import base64
import hashlib
import json
import logging.handlers
import os
import re
import shutil
import socket
import subprocess
import time
import warnings
from datetime import datetime

import requests
import xmltodict

from mastools_util import get_ns_version

log_file_name_local = os.path.basename(__file__)
LOG_FILENAME = '/var/mastools/logs/' + log_file_name_local + '.log'
LOG_MAX_BYTE = 10 * 1024 * 1024
LOG_BACKUP_COUNT = 20

TELEMETRY_POLL_TIME = 24*60*60

NETSCALER_MASTOOLS_DIR = '/var/mastools/'
NETSCALER_TELEMETRY_PACKAGE_DIR = NETSCALER_MASTOOLS_DIR+'telemetry/'
NETSCALER_TELEMETRY_DATA_DIR = NETSCALER_TELEMETRY_PACKAGE_DIR + 'executor/'
NETSCALER_TELEMETRY_FORWARDER_DIR = NETSCALER_TELEMETRY_PACKAGE_DIR + 'forwarder/'
NETSCALER_TELEMETRY_DOWNLOAD_DIR = NETSCALER_TELEMETRY_PACKAGE_DIR + 'download/'
NETSCALER_TELEMETRY_GATEWAY_DIR = NETSCALER_TELEMETRY_PACKAGE_DIR + 'gateway_data/adc/'
TELEMETRY_PACKAGE_FILE_NAME = "telemetry_package.tgz"
CA_CERT_CHAIN= '/var/mastools/cert/ca-chain.cert.pem'
CERT_BUNDLE_PATH='/var/mastools/cert/all_cacerts.pem'
if not os.path.exists(CERT_BUNDLE_PATH):
    CERT_BUNDLE_PATH = '/var/mastools/cert/cacert.pem'

DOWNLOAD_URL_PROD = "download.citrixnetworkapi.net"
DOWNLOAD_URL_STAGING = "download.citrixnetworkapistaging.net"

PROXY_CONF_FILE = "/var/mastools/proxy.conf"
AGENT_CONF_FILE = '/var/mastools/conf/agent.conf'
NS_CONFIG_FILE='/nsconfig/ns.conf'

TEST_PROPERTY_FILE = '/tmp/telemetry.properties'
MASTOOLS_PYTHON='/var/python/bin/python'

MAX_AGE_DAYS= 90

FAILURE_STEP_ELIGIBILITY_CHECK = "ELIGIBILITY_CHECK"
FAILURE_STEP_UBER_SCRIPT_EXECUTION = "UBER_SCRIPT_EXECUTION"
FAILURE_STEP_TLMY_SCRIPT_EXECUTION = "TLMY_SCRIPT_EXECUTION"
FAILURE_STEP_FWD_PAYLOAD = "FWD_PAYLOAD"

TELEMETRY_STATUS_CODE_SUCCESS = 0
TELEMETRY_STATUS_CODE_PARTIAL_FAILURE = 5

# Set up a specific logger with our desired output level
logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)

def log_namer(name):
    return name + ".gz"

def log_rotator(source, dest):
    with open(source, 'rb') as f_in:
        with gzip.open(dest, 'wb') as f_out:
            shutil.copyfileobj(f_in, f_out)
    os.remove(source)
#==============================override RotatingFileHandler==================================================
# this class overrides the doRollover() of Class RotatingFileHandler. 
# it will do following-
#    1. if corresponding zipped file(dstChkfile) is present,it will remove the chkFile.
#    2. if corresponding zipped file(dstChkFile) is not present, it will zip it.
 
class customLoghandler(logging.handlers.RotatingFileHandler):
    def doRollover(self):
        if self.backupCount > 0:
            for i in range(self.backupCount , 0, -1):
                chkFile = "{}.{}".format(self.baseFilename, i)
                dstChkFile ="{}.{}.gz".format(self.baseFilename, i )
                if os.path.exists(chkFile):
                    if os.path.exists(dstChkFile):
                        os.remove(chkFile)
                    else:
                        log_rotator(chkFile,dstChkFile)    
        super().doRollover()
#============================================================================================================
# Add the log message handler to the logger
logger_handler = customLoghandler(LOG_FILENAME, maxBytes=LOG_MAX_BYTE, backupCount=LOG_BACKUP_COUNT)
logger_handler.rotator = log_rotator
logger_handler.namer   = log_namer
logger_formatter = logging.Formatter(fmt='%(asctime)s:%(funcName)s:%(lineno)d: [%(levelname)s] %(message)s', datefmt="%Y-%m-%d %H:%M:%S")
logger_handler.setFormatter(logger_formatter)
logger.addHandler(logger_handler)
#======================================================================================================================================================


warnings.filterwarnings("ignore")
IS_PROXY_SET = ''

class mastools_tlmy():
    def __init__(self):
        self.ver = "2.0"
        self.agent_conf_dict = self.read_agent_conf()
        self.config_file_path = "/var/mastools/conf/nsconsole.conf"
        self.token = str(int(time.time()))
        self.evergreen_download_status = ""
        self.evergreen_download_error_message = ""
        self.failure_step = ""
        self.error_message = ""
        self.feature_telemetry_result = []
        self.static_nsconf_last_parse_time = 0
        self.static_self_ip_address = ""

    def read_agent_conf(self):
        agent_conf_dict = {}
        try:
            if os.path.exists(AGENT_CONF_FILE):
                try:
                    with open(AGENT_CONF_FILE, "r+") as agent_conf_file:
                        content= agent_conf_file.read()
                        agent_conf_file.close()
                except Exception as e:
                    logger.error("Failed to read agent conf file. %s." %(str(e)))
                    return agent_conf_dict
            else:
                logger.error("agent.conf file not found.")
                return agent_conf_dict

            agent_conf_dict = xmltodict.parse(content)['mps_agent']
        except Exception as e:
            logger.error('Error in read_agent_conf  '+ str(e))
            return agent_conf_dict

        return agent_conf_dict


    def read_file(self,filename):
        try:
            file = open(filename, "r")
            return file.read()
        except Exception as e:
            logger.error(e)
            raise e

    def get_CurrentISO8601UtcTimeInString(self):
        return (str(datetime.now().isoformat()) + 'z')

    def sys_command_with_exit_code(self, *args):
        try:
            logger.debug("Executing command: %s", args)

            p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            # Capture both stdout and stderr
            stdout, stderr = p.communicate()

            exit_code = p.returncode

            # Decode outputs
            stdout = stdout.decode().strip() if stdout else ""
            stderr = stderr.decode().strip() if stderr else ""

            if exit_code != 0:
                logger.error("Command failed with exit code %d: %s", exit_code, stderr)

            return stdout, stderr, exit_code

        except Exception as e:
            logger.error("Error executing command: %s", str(e))
            raise

    def sys_command(self, *args):
        try:
            logger.debug(list(args))
            if not isinstance(args, list):
                args = list(args)
            p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            result, err = p.communicate()
            if p.returncode != 0:
                raise Exception("Command %s failed with error: %s" %(' '.join(args), err.decode()))
            return result.decode()
        except Exception as e:
            logger.error(str(e))
            raise e

    def getMyIpAddress(self):

        # get current time
        current_time = int(time.time())

        # find 95% of TELEMETRY_POLL_TIME
        ninety_five_percent = int(0.95 * TELEMETRY_POLL_TIME)

        # if NS config file is not parsed in last TELEMETRY_POLL_TIME, parse it again
        if current_time - self.static_nsconf_last_parse_time > (ninety_five_percent):
            logger.info("Parsing NS config file")
            ns_ip_address = '127.0.0.1'
            try:
                with open(NS_CONFIG_FILE) as f:
                    for line in f:
                        if (not line.startswith('#')):
                            if 'set' in line and 'ns' in line and 'config' in line and '-IPAddress' in line:
                                line = line.strip()
                                logger.info('NetScaler IP configuration line: ' + line)
                                split_text = re.split(r'\s+', line)
                                if((split_text[0] == 'set') and (split_text[1] == 'ns') and (split_text[2] == 'config') and (split_text[3] == '-IPAddress')):
                                    ns_ip_address = split_text[4]
                                    logger.info('NetScaler IP Address: ' + ns_ip_address)
                                break
            except Exception as e:
                logger.error('Exception in getMyIpAddress: ' + str(e))
                logger.error('Using default NetScaler IP Address: ' + ns_ip_address)

            self.static_self_ip_address = ns_ip_address
            self.static_nsconf_last_parse_time = current_time
        else:
            logger.info("Using cached IP address")
        return self.static_self_ip_address

    def get_certchainhash(self):
        try:
            with open(CA_CERT_CHAIN) as f:
                m  = hashlib.sha256()
                m.update((f.read()).encode('utf-8'))
                sha = m.digest()
            return base64.b64encode(sha).decode()
        except Exception as e:
            logger.error('Error in reading file '+ str(e))

    def get_property(self,property_file, key, default_value):
    # Create a ConfigParser object
        value = default_value
        try:
            if (os.path.exists(property_file)):
                with open(property_file) as f:
                    for line in f:
                        line = line.strip()
                        split_text = re.split(r'[ \t=]+', line)
                        if (key == split_text[0]):
                            value = split_text[1]
        except Exception as e:
            logger.error('Error in reading get_property '+ str(e))
        return value

    def get_product(self):
        return self.get_property(TEST_PROPERTY_FILE,'product','nstelemetry')

    def get_download_url(self):
        return self.get_property(TEST_PROPERTY_FILE,'url',DOWNLOAD_URL_PROD)

    def get_proxy_details(self):
        try:
            if not os.path.exists(PROXY_CONF_FILE):
                logger.debug('proxy.conf file is not present')
                self.is_proxy_set = False
                return None

            with open(PROXY_CONF_FILE, "r+") as proxy_conf_file:
                content = proxy_conf_file.read()


            PROXY_CONF_DICT = xmltodict.parse(content)['proxy']
            logger.debug("PROXY_CONF_DICT is : "+str(PROXY_CONF_DICT))

            if 'proxy_server_ip' not in PROXY_CONF_DICT.keys():
                logger.debug("proxy_server_ip field missing in proxy.conf")
                self.is_proxy_set = False
                return None

            PROXY_SERVER_IP = PROXY_CONF_DICT.get('proxy_server_ip')
            if PROXY_SERVER_IP is None:
                logger.debug("proxy_server_ip field in proxy.conf is empty")
                self.is_proxy_set = False
                return None

            if 'proxy_server_port' not in PROXY_CONF_DICT.keys():
                logger.debug("proxy_server_port field missing in proxy.conf")

            PROXY_PORT = PROXY_CONF_DICT.get('proxy_server_port')
            if PROXY_PORT is None:
                logger.debug("proxy_server_port field in proxy.conf is empty. Trying with default port 443")
                PROXY_PORT = "443"

            if 'proxy_user' not in PROXY_CONF_DICT.keys():
                logger.debug("proxy_user field missing in proxy.conf")

            PROXY_USER = PROXY_CONF_DICT.get('proxy_user')

            if 'proxy_password' not in PROXY_CONF_DICT.keys():
                logger.debug("proxy_password field missing in proxy.conf")

            if PROXY_CONF_DICT.get('proxy_password') is not None:
                PROXY_PASSWORD = self.decrypt(PROXY_CONF_DICT['proxy_password'], self.conn)
                if PROXY_PASSWORD is None:
                    logger.error("error while decrypting password; issue in db or config file. set proxy to false")
                    self.is_proxy_set = False
                    return None

            self.is_proxy_set = True
            if PROXY_USER is not None and PROXY_PASSWORD is not None:
                logger.debug("Connecting to proxy server using user authentication")
                https_proxy = "http://" + PROXY_USER + ":" + PROXY_PASSWORD + "@" + PROXY_SERVER_IP + ":" + PROXY_PORT
            else:
                logger.debug("Connecting to proxy server without authentication")
                https_proxy = "http://" + PROXY_SERVER_IP + ":" + PROXY_PORT

            PROXIES = {
                'https': https_proxy
            }
        
            return PROXIES
        
        except Exception as excp:
            logger.error("Exception while getting proxy details. exception message : "+str(excp))
            self.is_proxy_set = False
        
        return None
        
    def downloadFile(self, url, product, targetpath, filename):
        try:
            endpoint = 'https://'
            endpoint += url
            endpoint += '/root/download/v1/public/software?'
            endpoint += 'product='
            endpoint += product
            endpoint += '&build=' 
            endpoint += self.get_certchainhash()
            endpoint += '&model='
            endpoint += filename
            timeout = 60
            response = ''
            PROXIES = self.get_proxy_details()
            if PROXIES is not None and PROXIES !='':
                response = requests.get(endpoint, verify=CERT_BUNDLE_PATH, proxies=PROXIES, stream=True, timeout=timeout)
            else:
                response = requests.get(endpoint, verify=CERT_BUNDLE_PATH, stream=True, timeout=timeout)

            if (response.status_code != requests.codes.ok):
                logger.error("Download of %s file from Download service failed, status: %s" %(filename, str(response.status_code)))
                return -1, "Download of %s file from Download service failed, status: %s"%(filename, str(response.status_code))

            with open(targetpath +'/'+ filename, 'wb') as fw:
                    for chunk in response.iter_content(chunk_size=1024):
                        if chunk:
                            fw.write(chunk)
            return 0, ""
        except Exception as e:
            logger.error('Exception in func=downloadService')
            logger.error(e)
            return -1, "Exception in downloadService: "+str(e)

    def download_telemetry_package(self):
        try:
            with open(CA_CERT_CHAIN) as f:
                prduct = self.get_product()
                url = self.get_download_url()
                logger.debug('URL for download:'+url)
                logger.debug('product for download:'+prduct)
                download_metadata_file = NETSCALER_TELEMETRY_DOWNLOAD_DIR+'metadata.pem'
                current_metadata_file = NETSCALER_TELEMETRY_PACKAGE_DIR+'metadata.pem'
                
                logger.debug('Download Metadata file:'+download_metadata_file)
                download_status, download_error_message = self.downloadFile(url, prduct ,NETSCALER_TELEMETRY_DOWNLOAD_DIR,'metadata.pem')
                if(download_status ==0):
                    downloaded_metadata = self.sys_command('openssl', 'cms', '-verify', '-in', download_metadata_file,'-inform','PEM','-CAfile',CA_CERT_CHAIN)
                    logger.debug('Downloaded Metadata file:'+downloaded_metadata)    
                    downloaded_metadata_json = json.loads(downloaded_metadata)
                    logger.debug('Downloaded Metadata file hash:'+downloaded_metadata_json['HashValue'])
                    if ((os.path.exists(current_metadata_file)) and (os.path.exists(NETSCALER_TELEMETRY_PACKAGE_DIR+'telemetry_package.tgz'))):
                        current_metadata = self.sys_command('openssl', 'cms', '-verify', '-in', current_metadata_file,'-inform','PEM','-CAfile',CA_CERT_CHAIN)
                        logger.debug('Current Metadata file:'+current_metadata)
                        current_metadata_json = json.loads(current_metadata)
                        if(current_metadata_json['HashValue'] == downloaded_metadata_json['HashValue']):
                            logger.debug("Device already have latest metadata file")
                            self.evergreen_download_status = "version same"
                            return 0
                        else:
                             logger.debug("Download service has a newer version of metadata file")
                    
                    logger.debug('Download Telemetry package telemetry_package.tgz')
                    download_status, download_error_message = self.downloadFile(url, prduct, NETSCALER_TELEMETRY_DOWNLOAD_DIR,'telemetry_package.tgz')
                    if download_status == 0:
                        sha=''
                        with open(NETSCALER_TELEMETRY_DOWNLOAD_DIR + 'telemetry_package.tgz','rb') as f:
                            m  = hashlib.sha256()
                            m.update(f.read())
                            sha = m.digest()
                            downloaded_file_hash = base64.b64encode(sha).decode()
                            if(downloaded_file_hash != downloaded_metadata_json['HashValue']):
                                logger.debug('Downloaded file hash:'+ downloaded_file_hash +' is not same as metadata hash:'+downloaded_metadata_json['HashValue'])
                            else:
                                logger.debug('Copying downloaded files to '+NETSCALER_TELEMETRY_PACKAGE_DIR) 
                                self.sys_command('mv',NETSCALER_TELEMETRY_DOWNLOAD_DIR+'telemetry_package.tgz', NETSCALER_TELEMETRY_PACKAGE_DIR+'telemetry_package.tgz')
                                self.sys_command('mv',download_metadata_file,current_metadata_file)
                            self.evergreen_download_status = "success"
                            return 0
                    else:
                        logger.error('Error in downloading telemetry_package.tgz')
                        self.evergreen_download_status = "failure"
                        self.evergreen_download_error_message = "Error in downloading telemetry_package.tgz + "+download_error_message
                else:
                    logger.error('Error in downloading Metadata file')
                    self.evergreen_download_status = "failure"
                    self.evergreen_download_error_message = "Error in downloading Metadata file : "+download_error_message
                    return -1
                return 0
        except Exception as e:
            logger.error('Exception in download_telemetry_package'+str(e))
            self.evergreen_download_status = "failure"
            self.evergreen_download_error_message = "Exception in download_telemetry_package" + str(e)
            return -1


    def collect_telemetry(self, ccid=None):
        try:
            self.proxies = self.get_proxy_details() # None if proxy not set
            self.sys_command('mkdir', '-p', NETSCALER_TELEMETRY_DATA_DIR)
            self.sys_command('mkdir', '-p', NETSCALER_TELEMETRY_DOWNLOAD_DIR)
            self.download_telemetry_package()
            token = self.token
            logger.debug ('Starting Telemetry. Time: '+token)
            os.chdir(NETSCALER_TELEMETRY_DATA_DIR)
            
            logger.debug ('Extracting Telemetry package')
            self.sys_command('tar','-xvzf', NETSCALER_TELEMETRY_PACKAGE_DIR+TELEMETRY_PACKAGE_FILE_NAME, '-C', NETSCALER_MASTOOLS_DIR)
            logger.debug ('Running script telemetry_orchestrator.py')

            outstr, errstr, exit_code = self.sys_command_with_exit_code(MASTOOLS_PYTHON, NETSCALER_TELEMETRY_DATA_DIR+'telemetry_orchestrator.py','--mode','executePackage','--host','netscaler','--consoleFormFactor','service','--collector','mastools','--collectionMode','automatic','--runOptionalTelemetry','yes','--identityToken',token, '--ccid', ccid)

            if exit_code != 0:
                logger.error('Error in executing telemetry_orchestrator.py')
                # if errstr contains "unrecognized arguments: --ccid" then log it
                if "unrecognized arguments: --ccid" in errstr:
                    logger.error("telemetry package version is older. It does not support --ccid argument")
                else:
                    logger.error('Error in executing telemetry_orchestrator.py: ' + errstr)
                self.failure_step = FAILURE_STEP_UBER_SCRIPT_EXECUTION
                self.error_message = errstr
                return -1
            else:
                # redirect errs to outstr
                # this should not be needed but somehow telemetry_orchestrator.py is writing everything to stderr and nothing on stdout
                if errstr:
                    outstr += errstr
                logger.debug (outstr)


            logger.debug ('Running script telemetry_orchestrator.py completed')
            
            telemetry_file = NETSCALER_TELEMETRY_DATA_DIR+'output_'+token+'.tgz'
            telemetry_result_dir = NETSCALER_TELEMETRY_PACKAGE_DIR+'temp_data/output/telemetry_result/'
            telemetry_ready_dir = NETSCALER_TELEMETRY_PACKAGE_DIR+'ready_data/netscaler/'+self.getMyIpAddress()+'/'
            telemetry_status_file = NETSCALER_TELEMETRY_DATA_DIR + 'output/telemetry.status'
            if (os.path.exists(telemetry_status_file)):
                with open(telemetry_status_file) as f:
                    lines = f.readlines()
                    execution_status = ""
                    for line in lines:
                        if line.startswith("execution_status:"):
                            execution_status = line.split(":", 1)[1].strip()
                        elif line.startswith("error_message:"):
                            error_message = line.split(":", 1)[1].strip()
                    if int(execution_status) not in [TELEMETRY_STATUS_CODE_SUCCESS, TELEMETRY_STATUS_CODE_PARTIAL_FAILURE]:
                        logger.error('Telemetry script execution failed:'+execution_status)
                        self.failure_step = FAILURE_STEP_UBER_SCRIPT_EXECUTION
                        self.error_message = error_message if error_message else ""
                        return -1


            self.sys_command('rm', '-rf', NETSCALER_TELEMETRY_PACKAGE_DIR+'temp_data')
            self.sys_command('mkdir', '-p' , NETSCALER_TELEMETRY_PACKAGE_DIR+'temp_data/')
            self.sys_command('rm', '-rf', NETSCALER_TELEMETRY_PACKAGE_DIR+'ready_data')
            self.sys_command('mkdir', '-p', telemetry_ready_dir)
            
            if (os.path.exists(telemetry_file)):
                logger.debug ('Extracting telemetry result file '+telemetry_file)
                self.sys_command('tar', '-xvzf', telemetry_file, '-C', NETSCALER_TELEMETRY_PACKAGE_DIR + 'temp_data/')
                for file_name in os.listdir(NETSCALER_TELEMETRY_PACKAGE_DIR+'temp_data/output/telemetry_result/'):
                    if(os.path.isfile(telemetry_result_dir+file_name)):
                        logger.debug ('Copying file to ready data folder:'+file_name)
                        shutil.copy2(telemetry_result_dir+file_name,telemetry_ready_dir+file_name)
            else:
                self.failure_step = FAILURE_STEP_UBER_SCRIPT_EXECUTION
                self.error_message = "Telemetry not generated:"+telemetry_file
                logger.error('Telemetry not generated:'+telemetry_file)
                return -1
            return 0

        except Exception as e:
            logger.error('Exception in collect_telemetry '+str(e))
            self.failure_step = FAILURE_STEP_UBER_SCRIPT_EXECUTION
            self.error_message = "Exception in collect_telemetry:"+str(e)

            return -1

    def forward_telemetry(self):
        try:
            logger.debug('Forward Telemetry data to console service')
            self.sys_command(MASTOOLS_PYTHON, NETSCALER_TELEMETRY_FORWARDER_DIR+'tlmy_forwarder.py','--consoleFormFactor','service','--collector','mastools')
            logger.debug('Forward Telemetry data to console service completed')
            forwarder_result_file = NETSCALER_TELEMETRY_FORWARDER_DIR + 'tlmy_forwarder_insights.json'
            if (os.path.exists(forwarder_result_file)):
                with open(forwarder_result_file) as f:
                    data = json.load(f)
                    if data.get("overall_status") == "failure":
                        logger.error('Forward Telemetry failed: '+data.get("error_message"))
                        self.failure_step = FAILURE_STEP_FWD_PAYLOAD
                        self.error_message = data.get("error_message")
                    else:
                        device_info = next(iter(data.get('devices', {}).values()), {})
                        if device_info.get('status') == "failure":
                            logger.error('Forward Telemetry failed: '+device_info.get('error_message'))
                            self.failure_step = device_info.get('fwd_step') if device_info.get('fwd_step') else \
                                FAILURE_STEP_FWD_PAYLOAD
                            self.error_message = device_info.get('error_message')
                os.remove(forwarder_result_file)
            return "success", ""
        except Exception as e:
            logger.error('Exception in forward_telemetry '+str(e))
            self.failure_step = FAILURE_STEP_FWD_PAYLOAD
            self.error_message = "Exception in forward_telemetry:"+str(e)

    def become_telemetry_collector_for_this_device(self):
        # create directory /var/mastools/conf/ if it does not exist
        if not os.path.exists("/var/mastools/conf/"):
            os.makedirs("/var/mastools/conf/")

        self_ip_address = self.getMyIpAddress()
        current_time = int(time.time())
        dictionary = {
            "collector_ip": self_ip_address,
            "console_deployment": "service",
            "console_type": "mastools",
            "configuration_timestamp": current_time,
            "collector_port": "5140"
        }
        with open(self.config_file_path, 'w') as file:
            json.dump(dictionary, file)


    # Validate IP address
    def is_valid_ip(self, ip):
        try:
            # Check IPv4
            socket.inet_pton(socket.AF_INET, ip)
            return True
        except socket.error:
            pass

        try:
            # Check IPv6
            socket.inet_pton(socket.AF_INET6, ip)
            return True
        except socket.error:
            pass

        return False

    # Validate timestamp
    def is_valid_epoch(self, timestamp):
        try:
            int_ts = int(timestamp)
            # Ensure it is a reasonable epoch time
            return 0 <= int_ts <= 2147483647
        except ValueError:
            return False


    def should_telemetry_be_done(self):
        current_time = int(time.time())
        dictionary = {}

        if os.path.exists(self.config_file_path):
            try:
                with open(self.config_file_path, 'r') as file:
                    dictionary = json.load(file)

                # Validation
                if not isinstance(dictionary, dict):
                    raise ValueError("JSON content is not a valid dictionary.")

            except (ValueError, getattr(json, 'JSONDecodeError', ValueError)) as e:
                # log the error. log should contain filename as well as exception
                logger.error("Error while reading JSON file %s: %s", self.config_file_path, str(e))
                return False

            required_keys = ["collector_ip", "console_deployment", "console_type"]
            for key in required_keys:
                if key not in dictionary:
                    logger.error("Mandatory key: %s missing from JSON file %s", key, self.config_file_path)
                    return False

            if "configuation_timestamp" not in dictionary and "configuration_timestamp" not in dictionary:
                logger.error("Mandatory key: configuration_timestamp missing from JSON file %s", self.config_file_path)
                return False

            if not self.is_valid_ip(dictionary["collector_ip"]):
                logger.error("collector_ip: %s is not a valid ip address", dictionary["collector_ip"])
                return False

            if dictionary["console_deployment"] not in ["onprem", "service", "unmanaged"]:
                logger.error("console_deployment: %s is not a valid value", dictionary["console_deployment"])
                return False

            if dictionary["console_type"] not in ["onprem-server", "agent", "mastools", "unmanaged"]:
                logger.error("console_type: %s is not a valid value", dictionary["console_type"])
                return False

            if "configuation_timestamp" in dictionary:
                if not self.is_valid_epoch(dictionary["configuation_timestamp"]):
                    logger.error("configuation_timestamp: %s is not a valid epoch time", str(dictionary["configuation_timestamp"]))
                    return False

            if "configuration_timestamp" in dictionary:
                if not self.is_valid_epoch(dictionary["configuration_timestamp"]):
                    logger.error("configuration_timestamp: %s is not a valid epoch time", str(dictionary["configuration_timestamp"]))
                    return False
        else:
            logger.info("Configuration file not found. Lets take over this device to perform telemetry")
            self.become_telemetry_collector_for_this_device()
            return True

        # Handle empty dictionary
        if not dictionary:
            logger.info("Dictionary is empty. Lets take over this device to perform telemetry")
            self.become_telemetry_collector_for_this_device()

        else:
            if "configuation_timestamp" in dictionary:
                config_ts = int(dictionary["configuation_timestamp"])
            else:
                config_ts = int(dictionary["configuration_timestamp"])

            self_ip_address = self.getMyIpAddress()

            if current_time - config_ts >= 86400:  # 24 hours = 86400 seconds
                logger.info("Configuration is older than 24 hours. Taking over this device to perform telemetry.")
                self.become_telemetry_collector_for_this_device()

            elif current_time - config_ts < 86400 and \
                    dictionary["collector_ip"] == self_ip_address and \
                    dictionary["console_deployment"] == "service" and \
                    dictionary["console_type"] == "mastools":
                logger.info("Device is actively managed by self. Updating configuration timestamp.")
                self.become_telemetry_collector_for_this_device()
            else:
                logger.info("Device is actively managed by another console: %s. Nothing to be done.", dictionary["collector_ip"])
                return False
        return True

    def get_agent_and_telemetry_version(self):
        try:
            with open('/var/mastools/version.txt', 'r') as file:
                agent_version = file.read().strip()
        except IOError as e:
            logger.error('Not able to read agent version from /var/mastools/version.txt error={}'.format(repr(e)))
            agent_version = ""

        try:
            with open('/var/mastools/telemetry/version.conf', 'r') as file:
                telemetry_version = file.read().strip()
        except IOError as e:
            logger.error('Not able to read telemetry version from /var/mastools/telemetry/version.conf error={}'.
                         format(repr(e)))
            telemetry_version = ""
        return agent_version, telemetry_version

    def extract_feature_telemetry_status(self):
        telemetry_status_file = NETSCALER_TELEMETRY_DATA_DIR + 'output/ns_telemetry_config_schema_result.json'
        if (os.path.exists(telemetry_status_file)):
            with open(telemetry_status_file) as f:
                data = json.load(f)
                for script in data.get('telemetryScripts', []):
                    telemetry_info = {
                        'name': script.get('telemetryName'),
                        'telemetry_status': script.get('executionResult', {}).get('status'),
                        'failure_step': "",
                        'error_message': ""
                    }
                    if script.get('executionResult', {}).get('status') != "success":
                        telemetry_info['failure_step'] = "TLMY_SCRIPT_EXECUTION"
                        telemetry_info['error_message'] = script.get('executionResult', {}).get('message')

                    self.feature_telemetry_result.append(telemetry_info)

    def forward_telemetry_insights(self):
        try:
            logger.debug('Forwarding Telemetry insights to console service')
            agent_version, telemetry_version = self.get_agent_and_telemetry_version()
            insights = {
                "cc_id": self.agent_conf_dict['customerid'],
                "agent_version": agent_version,
                "agent_ip": "",
                "token": self.token,
                "device_ip": self.getMyIpAddress(),
                "device_version": get_ns_version(),
                "managed_by": "mastools",
                "telemetry_package_version": telemetry_version,
                "insights_time_stamp":str(int(time.time())),
                "evergreen_download_status": self.evergreen_download_status,
                "evergreen_download_error_details": self.evergreen_download_error_message,
                "device_telemetry_status" : "success" if not self.failure_step else "failure",
                "failure_step": self.failure_step,
                "error_message": self.error_message,
                "results": self.feature_telemetry_result
            }
            # write this json to a file
            with open(NETSCALER_TELEMETRY_PACKAGE_DIR + 'tlmy_insights.json', 'w') as f:
                json.dump(insights, f)
            logger.info("Telemetry insights written to file: %s",NETSCALER_TELEMETRY_PACKAGE_DIR + 'tlmy_insights.json')
        except Exception as e:
            logger.error('Exception in forward_telemetry_insights '+str(e))

def main():
    parser = argparse.ArgumentParser(description="MASTools Telemetry")
    parser.add_argument("--daemon", choices=["yes", "no"], help="Run the script as daemon")
    args = parser.parse_args()

    run_as_daemon = True
    if (args.daemon is not None) :
        if(args.daemon == 'no'):
            run_as_daemon = False
    
    first_iteration = True
    while True:
        ccid = ""
        tlmy_object = None
        try:
            logger.info("---------------- Starting Telemetry ----------------")
            if not os.path.exists(NETSCALER_TELEMETRY_GATEWAY_DIR):
                os.makedirs(NETSCALER_TELEMETRY_GATEWAY_DIR)

            initial_wait = 10*60
            if(os.path.isfile(TEST_PROPERTY_FILE)):
                initial_wait = 60

            if first_iteration == True:
                logger.info("Sleeping initially for " + str(initial_wait) +" secs")
                time.sleep(initial_wait)
            first_iteration = False

            tlmy_object = mastools_tlmy()
            ccid = tlmy_object.agent_conf_dict.get("customerid")
            if not ccid:
                logger.error("customerid is not present in agent.conf file. Continuing.")
            else:
                logger.info("Checking if telemetry should be done on this device or not")
                if tlmy_object.should_telemetry_be_done() == False:
                    logger.info("Telemetry on this device is being managed by another console. Nothing to be done")
                    tlmy_object.failure_step = FAILURE_STEP_ELIGIBILITY_CHECK
                    tlmy_object.error_message = "Telemetry on this device is being managed by another console."
                else:
                    logger.info("This device is actively managed by self. Proceeding with telemetry")
                    logger.info('Collecting Telemetry data')
                    tlmy_object.collect_telemetry(ccid)
                    if not tlmy_object.failure_step:
                        logger.info('Collection of Telemetry data completed')
                        tlmy_object.forward_telemetry()
                        if not tlmy_object.failure_step:
                            tlmy_object.extract_feature_telemetry_status()
                    else:
                        logger.error('Error in collecting telemetry data')
        except Exception as e:
            logger.error('Exception: ' + str(e))
            if tlmy_object:
                tlmy_object.failure_step = FAILURE_STEP_UBER_SCRIPT_EXECUTION
                tlmy_object.error_message = "Exception in main script: "+str(e)
        if ccid and tlmy_object:
            tlmy_object.forward_telemetry_insights()

        if run_as_daemon:
            logger.info("Sleeping for " + str(TELEMETRY_POLL_TIME) + " secs")
            time.sleep(TELEMETRY_POLL_TIME)
        else:    
            break


if __name__ == '__main__':
    main()

