#! /usr/bin/env python

# Copyright 2025-2025 Citrix Systems, Inc. All rights reserved.
# This software and documentation contain valuable trade
# secrets and proprietary property belonging to Citrix Systems, Inc.
# None of this software and documentation may be copied,
# duplicated or disclosed without the express
# written permission of Citrix Systems, Inc.

"""
Automate review of code changes via user defined checks.

Dependency packages: None
"""
import json
import sys

# Allowed locations per key with "payload" instead of "body"
allowed_locations_map = {
    "Auth-Token": ["header"],
    "HostName": ["header", "payload"],
    "Content-type": ["header"],
    "User-agent": ["header"],
    "Role": ["header", "payload"],
    "MachineIP": ["header", "payload"],
    "MachineName": ["header", "payload"],
    "ClusterTimestamp": ["payload"],
    "LogLevel": ["payload"],
    "MessageContent": ["payload"],
    "MessageName": ["payload"],
    "MessageSequenceNumber": ["payload"],
    "ModuleName": ["payload"],
    "Name": ["payload"],
    "NodeId": ["payload"],
    "PPE-coreId": ["payload"],
    "Partition": ["payload"],
    "Time": ["payload"],
}

special_keys = {"Content-type", "User-agent", "Role"}

suggested_defaults = {
    "Content-type": "application/json",
    "User-agent": "AUDITLOGS/1.0",
    "Role": "GW"
}


def get_user_choice(prompt, choices, default=None):
    choices_str = '/'.join([c.upper() if c == default else c for c in choices])
    while True:
        response = input(f"{prompt} ({choices_str}): ").strip().lower()
        if not response and default is not None:
            return default
        if response in choices:
            return response
        print(f"Invalid input. Please choose from: {choices_str}")


def get_custom_key_name(original_key):
    alias = input(f"Enter the display name (alias) for '{original_key}' (press Enter to keep as is): ").strip()
    return alias if alias else original_key


def get_default_value_for_key(key, current_default=None):
    if current_default:
        change = get_user_choice(
            f"Default value for '{key}' is '{current_default}'. Do you want to change it?",
            ['y', 'n'], default='n'
        )
        if change == 'y':
            while True:
                value = input(f"Enter new default value for '{key}': ").strip()
                if value:
                    return value
                print("Default value cannot be empty.")
        return current_default
    else:
        wants = get_user_choice(
            f"Do you want to set a default value for '{key}'?",
            ['y', 'n'], default='n'
        )
        if wants == 'y':
            while True:
                value = input(f"Enter default value for '{key}': ").strip()
                if value:
                    return value
                print(" Default value cannot be empty.")
    return None


def get_delimiter_interactively():
    print("\nFormat Delimiter Options:\n1. \\n (newline)")
    choice = get_user_choice("Select delimiter", ['n'], default='')
    return '' if choice == '' else '\n'


def build_schema_interactively():
    header = {}
    payload = {}

    print("Schema Builder: Follow the prompts to define your schema.\n")

    for original_key, allowed_locations in allowed_locations_map.items():
        try:
            include = get_user_choice(f"Include '{original_key}'?", ['y', 'n'], default='y')
            if include == 'n':
                continue

            new_name = get_custom_key_name(original_key)

            if len(allowed_locations) == 1:
                location = allowed_locations[0]
                print(f"'{original_key}' can only go in '{location}'. Automatically selected.\n")
            else:
                location = get_user_choice(
                    f"Where should '{original_key}' be included?",
                    [loc.lower() for loc in allowed_locations],
                    default='header'
                )

            current_default = suggested_defaults.get(original_key)
            default = get_default_value_for_key(original_key, current_default) if original_key in special_keys else None

            if location == "header":
                entry = {"name": new_name}
                if default is not None:
                    entry["value"] = default
                header[original_key] = entry
            elif location == "payload":
                payload[original_key] = new_name

        except Exception as e:
            print(f"Error while processing '{original_key}': {e}")
            continue

    print("\nFormat Options:")
    delimiter = get_delimiter_interactively()

    event_wrapper_choice = get_user_choice(
        "Do you want to enable the 'EventWrapper'? (This will prefix the output with 'EVENT: ' for structured logging.)",
        ['y', 'n'], default='y'
    )
    event_wrapper_enabled = "true" if event_wrapper_choice == 'y' else "false"
    #event_wrapper_enabled = True if event_wrapper_choice == 'y' else False

    format_obj = {"delimiter": delimiter, "eventWrapperEnabled": event_wrapper_enabled}

    return {"Header": header, "Payload": payload, "Format": format_obj}


def build_schema_with_defaults():
    header = {}
    payload = {}

    for original_key, allowed_locations in allowed_locations_map.items():
        alias = original_key
        default_value = suggested_defaults.get(original_key)

        entry = {"name": alias}
        if default_value is not None:
            entry["value"] = default_value

        if len(allowed_locations) == 1:
            location = allowed_locations[0]
        else:
            location = "header"

        if location == "header":
            header[original_key] = entry
        elif location == "payload":
            payload[original_key] = alias

    format_obj = {
        "delimiter": '',  # default delimiter in default mode updated here
        "eventWrapperEnabled": "true" # EventWrapper is enabled by default in default mode
    }

    return {"Header": header, "Payload": payload, "Format": format_obj}


def print_schema(schema):
    try:
        print(json.dumps(schema, indent=4))
    except Exception as e:
        print(f"Failed to print schema: {e}")


def save_schema_to_file(schema, filename=None):
    if filename:
        if not filename.endswith(".json"):
            filename += ".json"
        try:
            with open(filename, 'w', encoding='utf-8') as f:
                json.dump(schema, f, indent=4)
            print(f"Schema saved to '{filename}'")
        except Exception as e:
            print(f"Failed to save file '{filename}': {e}")
        return

    save = get_user_choice("\nDo you want to save the schema to a JSON file?", ['y', 'n'], default='n')
    if save == 'y':
        while True:
            file_name = input("Enter file name (without extension): ").strip()
            if not file_name:
                print("File name cannot be empty.")
                continue
            if not file_name.endswith(".json"):
                file_name += ".json"

            try:
                with open(file_name, 'w', encoding='utf-8') as f:
                    json.dump(schema, f, indent=4)
                print(f"Schema saved to '{file_name}'")
                break
            except Exception as e:
                print(f"Failed to save file: {e}. Try a different name or path.")


def main():
    try:
        args = sys.argv[1:]

        if '--help' in args:
            print("""\
Usage:
  python build_schema.py [--defaults] [--save <filename>] [--help]

Options:
  --defaults           Generate schema using default values without prompting.
                      - All fields are included.
                      - Original key is used as alias.
                      - Location defaults to "header" if multiple allowed.
                      - Default values are pre-filled for special keys:
                          - User-agent: "AUDITLOGS/1.0"
                          - Role: "GW"
                          - Content-type: "application/json"

  --save <filename>   Save the generated schema directly to a file without prompting.
                      The .json extension is added automatically if omitted.

  --help              Show this help message and exit.

Examples:
  python build_schema.py
      Launches interactive schema builder with prompts.

  python build_schema.py --defaults
      Automatically generates schema using default values.

  python build_schema.py --defaults --save audit_schema.json
      Generates default schema and saves it to 'audit_schema.json' without prompts.

  python build_schema.py --save my_schema.json
      Launches interactive mode and saves final schema to 'my_schema.json' after generation.
""")
            return

        use_default = '--defaults' in args
        save_file = None

        if '--save' in args:
            try:
                save_index = args.index('--save')
                if save_index + 1 >= len(args):
                    print("Missing filename after '--save'")
                    return
                save_file = args[save_index + 1]
            except IndexError:
                print("Missing filename after '--save'")
                return

        schema = build_schema_with_defaults() if use_default else build_schema_interactively()

        print_schema(schema)
        save_schema_to_file(schema, filename=save_file)

    except KeyboardInterrupt:
        print("\nProcess interrupted by user. Exiting.")
    except Exception as e:
        print(f"\nUnexpected error: {e}")


if __name__ == "__main__":
    main()


