#!/usr/bin/python
########################################################################################
#
#  Copyright (c) 2019-2023 Citrix Systems, Inc.
#  All rights reserved.
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions are met:
#      * Redistributions of source code must retain the above copyright
#        notice, this list of conditions and the following disclaimer.
#      * Redistributions in binary form must reproduce the above copyright
#        notice, this list of conditions and the following disclaimer in the
#        documentation and/or other materials provided with the distribution.
#      * Neither the name of the Citrix Systems, Inc. nor the
#        names of its contributors may be used to endorse or promote products
#        derived from this software without specific prior written permission.
#
#  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
#  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
#  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
#  DISCLAIMED. IN NO EVENT SHALL CITRIX SYSTEMS, INC. BE LIABLE FOR ANY
#  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
#  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
#  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
#  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
#  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
#  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
########################################################################################

"""
Utility functions for GCP
"""
from libnitrocli import nitro_cli, nitro_cli_output_parser
from google_compute_engine import metadata_watcher
from googleapiclient import discovery
from oauth2client.client import GoogleCredentials
from enum import Enum
from src.ha_state import HaStatus
import time
import os
import json
import signal
import logging
import socket
from src.ha_state import HaStatus
from src.ns_ha import NS
from netaddr import IPNetwork, IPAddress

RAIN_SCALE_BIN = "/var/python/bin/rain_scale"
RAIN_SCALE_PID_FILE = "/nsconfig/.GCP/rain_scale.pid"
class HaConfiguration(Enum):
    """Enum class for HA Configurations"""
    HA_NOT_CONFIGURED = 1
    HA_WITH_EIP = 2
    HA_WITH_PIP = 3
    HA_WITH_EIP_PIP = 4
    HA_CLUSTER_CLIP_WITH_PIP = 5

IAM_PERMS_HA_NOT_CONFIGURED = [
    "compute.forwardingRules.list",
    "compute.forwardingRules.setTarget",
    "compute.instances.get",
    "compute.instances.list",
    "compute.instances.setMetadata",
    "compute.targetInstances.create",
    "compute.targetInstances.list",
    "compute.targetInstances.use",
    "compute.zones.list",
]

IAM_PERMS_HA_WITH_EIP = [
    "compute.addresses.use",
    "compute.forwardingRules.list",
    "compute.forwardingRules.setTarget",
    "compute.instances.addAccessConfig",
    "compute.instances.deleteAccessConfig",
    "compute.instances.get",
    "compute.instances.list",
    "compute.instances.setMetadata",
    "compute.networks.useExternalIp",
    "compute.subnetworks.useExternalIp",
    "compute.targetInstances.create",
    "compute.targetInstances.list",
    "compute.targetInstances.use",
    "compute.zones.list",
    "compute.zoneOperations.get",
]

IAM_PERMS_HA_WITH_PIP = [
    "compute.forwardingRules.list",
    "compute.forwardingRules.setTarget",
    "compute.instances.get",
    "compute.instances.list",
    "compute.instances.setMetadata",
    "compute.instances.updateNetworkInterface",
    "compute.targetInstances.create",
    "compute.targetInstances.list",
    "compute.targetInstances.use",
    "compute.zones.list",
    "compute.zoneOperations.get",
]

IAM_PERMS_HA_WITH_EIP_PIP = [
    "compute.addresses.use",
    "compute.forwardingRules.list",
    "compute.forwardingRules.setTarget",
    "compute.instances.addAccessConfig",
    "compute.instances.deleteAccessConfig",
    "compute.instances.get",
    "compute.instances.list",
    "compute.instances.setMetadata",
    "compute.instances.updateNetworkInterface",
    "compute.networks.useExternalIp",
    "compute.subnetworks.useExternalIp",
    "compute.targetInstances.create",
    "compute.targetInstances.list",
    "compute.targetInstances.use",
    "compute.zones.list",
    "compute.zoneOperations.get",
]

IAM_PERMS_HA_CLUSTER_CLIP_WITH_PIP = [
    "compute.forwardingRules.list",
    "compute.forwardingRules.setTarget",
    "compute.instances.get",
    "compute.instances.list",
    "compute.instances.setMetadata",
    "compute.targetInstances.create",
    "compute.targetInstances.list",
    "compute.targetInstances.use",
    "compute.zones.list",
    "compute.addresses.list",
    "compute.instances.updateNetworkInterface",
    "compute.zoneOperations.get",
]

REQUIRED_IAM_PERMS = {
    HaConfiguration.HA_NOT_CONFIGURED: IAM_PERMS_HA_NOT_CONFIGURED,
    HaConfiguration.HA_WITH_EIP: IAM_PERMS_HA_WITH_EIP,
    HaConfiguration.HA_WITH_PIP: IAM_PERMS_HA_WITH_PIP,
    HaConfiguration.HA_WITH_EIP_PIP: IAM_PERMS_HA_WITH_EIP_PIP,
    HaConfiguration.HA_CLUSTER_CLIP_WITH_PIP: IAM_PERMS_HA_CLUSTER_CLIP_WITH_PIP,
}

IAM_NOT_OK_FILE = '/flash/nsconfig/.GCP/iam_not_ok'

#########################################################################
### XXX FreeBSD 11.x: socket.getaddrinfo() prefers IPV6 by default on
###                   11.x. Wrap socket.getaddrinfo() to use IPV4 instead
old_getaddrinfo = socket.getaddrinfo
def new_getaddrinfo(host, port, family=0, type=0, proto=0, flags=0):
    if family == socket.AF_UNSPEC:
        family = socket.AF_INET
    return old_getaddrinfo(host, port, family, type, proto, flags)
socket.getaddrinfo = new_getaddrinfo
##########################################################################

class GcpInstanceInfo():
    """ Contains Clould specific API. Must implement
        update_ha_info ,handle_failover and __init__"""

    def __init__(self, ns_obj):
        self.peer_ip = None
        self.network = None
        self.state = ns_obj.state
        """ TODO store inc_mode as boolean instead of string
        inc_mode required to determine whether migration of
        alias ip should be performed or not"""
        self.inc_mode = ns_obj.inc_mode
        self.project = None
        self.name = None
        self.zone = None
        self.network_interfaces = None
        self.peer_project = None
        self.peer_name = None
        self.peer_zone = None
        self.ip = None
        self.nitro = None
        self.management_nic = 0

        self.gather_self_info()
        self.init_nitro()
        if self.state not in [HaStatus.CCO, HaStatus.NON_CCO]:	
            if self.nitro is not None and self.update_management_info() is True:
                if ns_obj.peer_ip:
                    self.gather_peer_info(ns_obj.peer_ip, self.project)
            else:
                logging.error("Failed to get management nic details")

    def init_nitro(self):
        """ Initialize nitrocli object """
        try:
            self.nitro = nitro_cli()
            self.parser = nitro_cli_output_parser()
        except:
            self.nitro = None
            self.parser = None
            logging.error('Nitro cli config failed')

    def gather_self_info(self):
        """ Gather only data that remains same for the entire duration in
        instance """

        #TODO what if metadata server is down.
        #GCP does not throw any exception Nor it returns error
        #currently it crashes the thread without throwing any exception
        #We need to check with GCP on how to handle it properly
        data = metadata_watcher.MetadataWatcher().GetMetadata()
        # Format of data: 
        #{u'instance': {u'attributes': {},
        #       u'cpuPlatform' xxxxx,
        #       u'description' xxxxx,
        #       u'disks': [{u'deviceName' xxxxx,
        #                   u'index' xxxxx,
        #                   u'mode' xxxxx,
        #                   u'type' xxxxx,
        #       u'guestAttributes' xxxxx,
        #       u'hostname' xxxxx,
        #       u'id' xxxxx,
        #       u'image' xxxxx,
        #       u'machineType' xxxxx,
        #       u'maintenanceEvent' xxxxx,
        #       u'name' xxxxx,
        #       u'networkInterfaces': [{u'accessConfigs': [{u'externalIp' xxxxx,
        #                                                   u'type' xxxxx,
        #                               u'dnsServers': [u'169.254.169.254'],
        #                               u'forwardedIps': [],
        #                               u'gateway' xxxxx,
        #                               u'ip' xxxxx,
        #                               u'ipAliases': [],
        #                               u'mac' xxxxx,
        #                               u'mtu' xxxxx,
        #                               u'network' xxxxx,
        #                               u'subnetmask' xxxxx,
        #                               u'targetInstanceIps': []},


        self.project = data["project"]["projectId"]
        self.name = data["instance"]["name"]
        project_numeric_id = data['project']['numericProjectId']
        self.zone = data["instance"]["zone"].replace('projects/' + str(project_numeric_id) + '/zones/', '')
        self.network_interfaces = data["instance"]["networkInterfaces"]
        logging.debug("Self info gathered")

    def update_management_info(self):
        hanode_info = self.nitro.get_hanode()
        self.ip = hanode_info['hanode'][0]['ipaddress']
        for network_interface in self.network_interfaces:
            if network_interface["ip"] == self.ip:
                self.network = network_interface["network"].split('/')[-1]
                return True
            self.management_nic = self.management_nic + 1
        return False

    def gather_peer_info(self, peer_ip, project):
        """ Gather Peer info if peer is valid """
        if peer_ip != self.peer_ip:
            logging.debug("Updating peer info as previous data is stale")
            self.peer_name = None
            self.find_peer_using_ip(project, peer_ip)
            if self.peer_name is not None:
                self.peer_ip = peer_ip
                self.peer_project = project
                logging.debug("Peer info Gathered")
            else :
                logging.error("Could not find peer based on PeerIP %s", peer_ip)
                self.peer_ip = None

    def wait_for_response(self, compute, project, zone, operation):
        """ Wait for ip addition /removal to happen """
        while True:
            result = compute.zoneOperations().get(
                project=project,
                zone=zone,
                operation=operation).execute()
            if result['status'] == 'DONE':
                if 'error' in result:
                    return False
                logging.debug("Done.")
                return True
            logging.debug('Waiting for operation to finish...')
            time.sleep(0.5)

#Format of instance
#{u'canIpForward' xxxxx,
# u'cpuPlatform' xxxxx,
# u'creationTimestamp' xxxxx,
# u'deletionProtection' xxxxx,
# u'description' xxxxx,
# u'disks': [{u'autoDelete' xxxxx,
#             u'boot' xxxxx,
#             u'deviceName' xxxxx,
#             u'guestOsFeatures': [{u'type' xxxxx,
#             u'index' xxxxx,
#             u'interface' xxxxx,
#             u'kind' xxxxx,
#             u'mode' xxxxx,
#             u'source' xxxxx,
#             u'type' xxxxx,
# u'id' xxxxx,
# u'kind' xxxxx,
# u'labelFingerprint' xxxxx,
# u'machineType' xxxxx,
# u'metadata' xxxxx,
# u'name' xxxxx,
# u'networkInterfaces': [{u'accessConfigs': [{u'kind' xxxxx,
#                                             u'name' xxxxx,
#                                             u'natIP' xxxxx,
#                                             u'networkTier' xxxxx,
#                                             u'type' xxxxx,
#                         u'fingerprint' xxxxx,
#                         u'kind' xxxxx,
#                         u'name' xxxxx,
#                         u'network' xxxxx,
#                         u'networkIP' xxxxx,
#                         u'subnetwork' xxxxx,
#                        {u'accessConfigs': [{u'kind' xxxxx,
#                                             u'name' xxxxx,
#                                             u'natIP' xxxxx,
#                                             u'networkTier' xxxxx,
#                                             u'type' xxxxx,
#                         u'fingerprint' xxxxx,
#                         u'kind' xxxxx,
#                         u'name' xxxxx,
#                         u'network' xxxxx,
#                         u'networkIP' xxxxx,
#                         u'subnetwork' xxxxx,
#                        {u'accessConfigs': [{u'kind' xxxxx,
#                                             u'name' xxxxx,
#                                             u'natIP' xxxxx,
#                                             u'networkTier' xxxxx,
#                                             u'type' xxxxx,
#                         u'fingerprint' xxxxx,
#                         u'kind' xxxxx,
#                         u'name' xxxxx,
#                         u'network' xxxxx,
#                         u'networkIP' xxxxx,
#                         u'subnetwork' xxxxx,
#                        {u'accessConfigs': [{u'kind' xxxxx,
#                                             u'name' xxxxx,
#                                             u'natIP' xxxxx,
#                                             u'networkTier' xxxxx,
#                                             u'type' xxxxx,
#                         u'fingerprint' xxxxx,
#                         u'kind' xxxxx,
#                         u'name' xxxxx,
#                         u'network' xxxxx,
#                         u'networkIP' xxxxx,
#                         u'subnetwork' xxxxx,
# u'scheduling' xxxxx,
#                 u'onHostMaintenance' xxxxx,
#                 u'preemptible' xxxxx,
# u'selfLink' xxxxx,
# u'serviceAccounts': [{u'email' xxxxx,
#                       u'scopes': [u'https://www.googleapis.com/auth/cloud-platform']}],
# u'startRestricted' xxxxx,
# u'status' xxxxx,
# u'tags': {u'fingerprint': u'xxxxx'},
# u'zone': u'https://www.googleapis.com/compute/v1/projects/xxxxx/zones/xxxxx'}
# 

    def find_peer_using_ip(self, project, peer_ip ):
        """ Find the Peer using Project and NSIP .
            Searches all instances in all zones in the region."""
        logging.debug("Finding peer using ip %s", peer_ip)
        credentials = GoogleCredentials.get_application_default()
        service = discovery.build('compute',
                                  'v1',
                                  credentials=credentials, cache_discovery=False)

        request_zone = service.zones().list(project=project)
        while request_zone is not None:
            try:
                response_project = request_zone.execute()
                for zone in response_project['items']:
                    #Skip if current zone is not in same region
                    if not zone["description"].startswith(self.zone[:self.zone.rfind('-')]) :
                        logging.debug("Skipping zone - %s", zone["description"])
                        continue
                    request_list_instance = service.instances().list(project=project,
                                                                 zone=zone["description"])
                    while request_list_instance is not None:
                        response_list_instance = request_list_instance.execute()
                        if "items" in response_list_instance:
                            for instance in response_list_instance['items']:
                                if len(instance["networkInterfaces"]) > self.management_nic\
                                      and "networkIP" in instance["networkInterfaces"][self.management_nic]:
                                    if instance["networkInterfaces"][self.management_nic]["network"].rsplit('/', 1)[-1] != self.network:
                                        logging.debug("Skipping as network does not match %s", instance["networkInterfaces"][self.management_nic]["network"])
                                        continue
                                    if instance["networkInterfaces"][self.management_nic]["networkIP"] == peer_ip:
                                        logging.debug(instance)
                                        self.peer_name = instance["name"]
                                        self.peer_zone = zone["description"]
                                        logging.info("Found peer node %s in %s zone", self.peer_name, self.peer_zone)
                                        return
                        request_list_instance = service.instances().list_next(previous_request=request_list_instance,
                                                                           previous_response=response_list_instance)
                request_zone = service.zones().list_next(previous_request=request_zone,
                                                                   previous_response=response_project)
            except Exception as err:
                logging.exception("Encountered error while finding"
                                             "peer node:%s", str(err))
                return

    def get_vips(self):
        """Return the list of configured VIPs on primary"""
        if self.nitro is None:
            self.nitro = self.init_nitro()
        vips= []
        nsipdict = self.nitro.get_nsip()
        for ip in nsipdict['nsip']:
            if ip['type'] == 'VIP':
                vips.append(ip["ipaddress"])
        return vips

    def update_alias_ips(self, vips, peer_alias_ips, self_alias_ips=[]):
        """Returns true if alias IPs are configured as VIP on primary"""
        vip_found = False
        for vip in vips:
            for alias_ip in peer_alias_ips:
                if IPAddress(str(vip)) in IPNetwork(str(alias_ip["ipCidrRange"])) and alias_ip not in self_alias_ips:
                    logging.info("Alias IP configured as VIP - %s , moving to new primary",str(vip))
                    self_alias_ips.append(alias_ip)
                    peer_alias_ips.remove(alias_ip)
                    vip_found = True
        return vip_found

    #Should be called only if self.peer_ip is valid
    def migrate_alias_ips(self, service, self_interface, peer_interfaces, vips) :
        """ During failover move all VIP Alias IPs from secondary to Primary """
        resp_name_list_attach = []
        loop_count = min(len(self_interface), len(peer_interfaces))
        for i in range(0, loop_count):
            if "aliasIpRanges" in peer_interfaces[i]:
                if self.inc_mode == "DISABLED":
                    #Restore support for old configuration
                    self_alias_ips = peer_interfaces[i]["aliasIpRanges"]
                    peer_alias_ips = []
                else:
                    peer_alias_ips = peer_interfaces[i]["aliasIpRanges"]
                    if "aliasIpRanges" in self_interface[i]:
                        self_alias_ips = self_interface[i]["aliasIpRanges"]
                        vip_found = self.update_alias_ips(vips, peer_alias_ips, self_alias_ips)
                    else:
                        self_alias_ips = []
                        vip_found = self.update_alias_ips(vips, peer_alias_ips, self_alias_ips)
                    if vip_found is False:
                        logging.info("No VIPs found, skipping to next interface")
                        continue
                request_body = {"aliasIpRanges" :peer_alias_ips,
                         "fingerprint" : peer_interfaces[i]["fingerprint"]}
                request = service.instances().updateNetworkInterface(
                    project=self.peer_project,
                    zone=self.peer_zone,
                    instance=self.peer_name,
                    networkInterface=peer_interfaces[i]["name"],
                    body=request_body)
                try:
                    response = request.execute()
                    result = self.wait_for_response(service,
                                                self.peer_project,
                                                self.peer_zone,
                                                response["name"])
                    if result == False:
                        logging.error("Error while removing the alias ip config : %s."
                                          "Skip attach to other instance",
                                            str(peer_alias_ips))
                        continue
                except Exception as err:
                    logging.exception("Encountered error while removing "
                                                  "alias ip: %s", str(err))
                    continue
                request_body = {"aliasIpRanges" : self_alias_ips,
                                       "fingerprint" : self_interface[i]["fingerprint"]}
                request = service.instances().updateNetworkInterface(
                                 project=self.project,
                                 zone=self.zone,
                                 instance=self.name,
                                 networkInterface=self_interface[i]["name"],
                                 body=request_body)
                try:
                    response = request.execute()
                    resp_name_list_attach.append((i, response["name"]))
                except Exception as err:
                    logging.exception("Encountered error while adding "
                                                    "alias ip: %s", str(err))
        for i, req_name in resp_name_list_attach:
            result = self.wait_for_response(service,
                                            self.project,
                                            self.zone,
                                            req_name)
            if result == False:
                logging.error("Error while adding the alias ip config : %s."
                                        ,str(self_alias_ips))
                #TODO reattach to secondary? Whatif reattach also fails?

    def get_target_instances(self, service, instance_name, project, zone):
        "Return list of target-instances for an instance"
        target_tags  = []
        #Boolean variable to indicate target-instance created by VIP-Scaling
        adc_internal_target = False
        request = service.targetInstances().list(project=project, zone=zone)
        while request is not None:
            try:
                response = request.execute()
                if "items" in response:
                    for target_instance in response['items']:
                        target_instance_name = target_instance['instance'].split('/')[-1]
                        target_name = target_instance['selfLink'].split('/')[-1]
                        if target_instance_name == instance_name:
                            if target_name == (instance_name + "-adcinternal"):
                                adc_internal_target = True
                                continue
                            target_tags.append(target_name)
            except Exception as err:
                logging.exception("Could not get target instances info: %s", err)
            request = service.targetInstances().list_next(previous_request=request, previous_response=response)
        #Append target-tag created by VIP-Scaling at the end
        if adc_internal_target == True:
            target_tags.append(instance_name + "-adcinternal")
        return target_tags

    def update_forwarding_rules(self, service):
        "Check and Update target instances of forwarding rules"
        peer_target_tags = self.get_target_instances(service, self.peer_name,
                                         self.peer_project, self.peer_zone)
        if not peer_target_tags:
            logging.info("No target instances found, skipping update of forwarding rules")
            return
        logging.info("Peer target instances: %s", peer_target_tags)
        self_target_tags = self.get_target_instances(service, self.name,
                                                     self.project, self.zone)
        logging.info("Self target instances %s", self_target_tags)

        #If peer node has internal ADC target instace and self node does not then create for self node 
        if (self.peer_name+"-adcinternal") in peer_target_tags and (self.name+"-adcinternal") not in self_target_tags:
            if self.create_target_instance(service):
                self_target_tags.append(self.name+"-adcinternal")
                #Add sleep for created target instance to be ready
                time.sleep(2)

        if len(self_target_tags) != len(peer_target_tags):
            logging.error("Number of target instances for primary and secondary nodes do not match,"
                          "terminating update of forwarding rules")
            return
        region =  self.zone.split('-')[0] + "-" + self.zone.split('-')[1]
        request = service.forwardingRules().list(project=self.project, region=region)
        while request is not None:
            try:
                response = request.execute()
                if "items" in response:
                    for forwarding_rule in response["items"]:
                        for (self_target_tag, peer_target_tag) in zip(self_target_tags, peer_target_tags):
                            if "target" in forwarding_rule and forwarding_rule["target"].split('/')[-1] == peer_target_tag:
                                logging.info("Updating forwarding rule: %s", forwarding_rule["selfLink"].split('/')[-1])
                                target_link = forwarding_rule['target'].replace(self.peer_zone, self.zone)
                                target_link = target_link.replace(peer_target_tag, self_target_tag)
                                target_reference_body = {
                                    "target": target_link
                                }
                                request = service.forwardingRules().setTarget(project=self.project,
                                            region=region, forwardingRule=forwarding_rule['name'], body=target_reference_body)
                                response = request.execute()
            except Exception as err:
                logging.exception("Error in updating forwarding rules : %s", err)
                return
            request = service.forwardingRules().list_next(previous_request=request, previous_response=response)

    #Should be called only if self.peer_ip is valid
    def migrate_ip_from_peer(self) :
        """ Migrate all ips from new secondary to new primary  """
        logging.debug("Moving ip from secondary to primary")
        credentials = GoogleCredentials.get_application_default()
        service = discovery.build('compute',
                                  'v1',
                                  credentials=credentials, cache_discovery=False)

        self_interface = service.instances().get(
            project=self.project,
            zone=self.zone,
            instance=self.name).execute()["networkInterfaces"]

        try:
            peer_interfaces = service.instances().get(
                project=self.peer_project,
                zone=self.peer_zone,
                instance=self.peer_name).execute()["networkInterfaces"]
        except Exception as err :
            logging.error("Peer resource not found. Skipping ip movement")
            logging.error("Exception : %s", str(err))
            return
        loop_count = min(len(self_interface), len(peer_interfaces))
        vips = self.get_vips()
        self.migrate_alias_ips(service, self_interface, peer_interfaces,vips)
        configs = []
        for i in range(0, loop_count):
            if i == self.management_nic:
                continue
            if "accessConfigs" in peer_interfaces[i] \
                and len(peer_interfaces[i]["accessConfigs"]) > 0 and peer_interfaces[i]["networkIP"] in vips:
                for config in peer_interfaces[i]["accessConfigs"]:
                    if "natIP" in config:
                        request = service.instances().deleteAccessConfig(
                                    project=self.peer_project,
                                    zone=self.peer_zone,
                                    instance=self.peer_name,
                                    accessConfig=config["name"],
                                    networkInterface=peer_interfaces[i]["name"])
                        try:
                            response = request.execute()
                            configs.append((i, config, response["name"]))
                        except Exception as err:
                            logging.exception("Encounterd error while removing "
                                              "EIP: %s", str(err))
                        break
        self_pending_requests = []
        for i_tupple in configs:
            i, config, request_name = i_tupple

            # check if pending request is complete
            result = self.wait_for_response(service,
                                           self.peer_project,
                                           self.peer_zone,request_name )
            if result == False:
                logging.error("Failed to remove EIP. Moving-on to next EIP")
                continue

            request = service.instances().addAccessConfig(
                        project=self.project,
                        zone=self.zone,
                        instance=self.name,
                        networkInterface=self_interface[i]["name"],
                        body=config)
            try:
                response = request.execute()
                self_pending_requests.append(response["name"])
            except Exception as err:
                logging.exception("Encounterd error while "
                                            "adding EIP: %s", str(err))
        for resp in self_pending_requests:
            result = self.wait_for_response(service,
                                           self.project,
                                           self.zone,
                                           resp)
            if result == False :
                logging.error("Failed to attach EIP. Moving-on to next EIP")
        self.update_forwarding_rules(service)
    
    def perform_metadata_sync(self):
        """Sync VIP Scaling custom metadata on failover from primary to secondary"""
        credentials = GoogleCredentials.get_application_default()
        service = discovery.build('compute', 'v1', credentials=credentials, cache_discovery=False)
        self_metadata = None
        peer_metadata_vip = None
        peer_metadata = None
        request = service.instances().get(project=self.peer_project, zone=self.peer_zone,
                                            instance=self.peer_name)
        vip_key_present = False
        try:
            response = request.execute()
            if "items" in response['metadata']:
                peer_metadata = response['metadata']['items']
                counter = len(peer_metadata)
                for i in range(0, counter):
                    if "vips" in peer_metadata[i]['key'] :
                        peer_metadata_vip = peer_metadata[i]
                        peer_metadata[i] = None
                        vip_key_present = True
                        break
            if vip_key_present == True:
                peer_metadata_body={
                    "fingerprint": response['metadata']['fingerprint'],
                    "items": peer_metadata
                    }
                request = service.instances().setMetadata(project=self.peer_project, zone=self.peer_zone,
                                                            instance=self.peer_name, body=peer_metadata_body)
                response = request.execute()
        except Exception as err:
            logging.exception("Error in removing custom metadata from old primary: %s", err)
            return
        if peer_metadata_vip:
            request = service.instances().get(project=self.project, zone=self.zone, instance=self.name)
            vip_key_present = False
            try:
                response = request.execute()
                if "items" in response['metadata']:
                    self_metadata = response['metadata']['items']
                    counter = len(self_metadata)
                    for i in range(0, counter):
                        if "vips" in self_metadata[i]['key'] :
                            self_metadata[i]= peer_metadata_vip
                            vip_key_present = True
                            break
                    if vip_key_present is False:
                        self_metadata.append(peer_metadata_vip)
                else:
                    self_metadata = []
                    self_metadata.append(peer_metadata_vip)
                self_metadata_body={
                    "fingerprint": response['metadata']['fingerprint'],
                    "items": self_metadata
                }
                request = service.instances().setMetadata(project=self.project, zone=self.zone,
                                                        instance=self.name, body=self_metadata_body)
                response = request.execute()
            except Exception as err:
                logging.exception("Error in updating custom metadata of new primary: %s", err)

    def check_target_instance(self,service):
        """ Check if internal ADC target instance exists for self node """
        target_instance_name = self.name + '-adcinternal'
        adc_internal_target = False
        request = service.targetInstances().list(project= self.project, zone= self.zone)
        while request is not None:
            try:
                response = request.execute()
                for target_instance in response['items']:
                    if target_instance['instance'].split('/')[-1] == self.name and target_instance['selfLink'].split('/')[-1] == target_instance_name:
                        logging.info("ADC Target Instance exists")
                        adc_internal_target = True
                        break
            except Exception as err:
                logging.exception("ADC Target instance info could not be retrieved: %s", err)
            request = service.targetInstances().list_next(previous_request=request, previous_response=response)
        if adc_internal_target == False:
            self.create_target_instance(service)

    def create_target_instance(self,service):
        """ Create internal ADC target instance for self node """
        target_instance_name = self.name + '-adcinternal'
        logging.info("Creating ADC Target Instance")
        target_instance_body = {
        "name": target_instance_name,
            "description": "target instance created fot Citrix ADC",
            "zone": self.zone,
            "natPolicy": "NO_NAT",
            "instance": "zones/"+self.zone+"/instances/"+self.name,
        }
        request = service.targetInstances().insert(project=self.project, zone=self.zone, body=target_instance_body)
        try:
            response = request.execute()
            return True
        except Exception as err:
            logging.exception("ADC Target instance could not be created: %s", err)
            return False


    #Called only if self.state == SECONDARY
    def check_vips_custom_metadata(self):
        """ Check peer custom metadata for vips key and add target instance for self node if present"""
        credentials = GoogleCredentials.get_application_default()
        service = discovery.build('compute', 'v1', credentials=credentials, cache_discovery=False)
        request = service.instances().get(project=self.peer_project, zone=self.peer_zone, instance=self.peer_name)
        try:
            response = request.execute()
            if "items" in response['metadata']:
                peer_metadata = response['metadata']['items']
                counter = len(peer_metadata)
                for i in range(0, counter):
                    if "vips" in peer_metadata[i]['key']:
                        logging.info("VIPS custom metadata configured on Primary")
                        self.check_target_instance(service)
                        return
        except Exception as err:
            logging.exception("Error getting peer custom metadata: %s", err)

    def check_iam(self, ha_configuration):
        """ Check instance IAM. Returns the Missing IAMs """
        credentials = GoogleCredentials.get_application_default()
        service = discovery.build('cloudresourcemanager', 'v1',
                                     credentials=credentials, cache_discovery=False)

        test_permissions_request = {
            "permissions": REQUIRED_IAM_PERMS[HaConfiguration[ha_configuration]]
        }

        request = service.projects().testIamPermissions(resource=self.project,
                                                        body=test_permissions_request)
        try:
            response = request.execute()
        except Exception as err:
            logging.error("No IAM permissions for testIamPermissions API")
            logging.error("Exception %s", str(err))
            return (set(REQUIRED_IAM_PERMS[HaConfiguration[ha_configuration]]))
        logging.debug(response)
        return (set(REQUIRED_IAM_PERMS[HaConfiguration[ha_configuration]]) - set(response["permissions"]))

    def check_rain_scale(self):
        NS.launch_process_if_not_running(RAIN_SCALE_BIN, RAIN_SCALE_PID_FILE)

    def signal_rain_scale(self):
        NS.send_signal_to_process(signal.SIGUSR1, RAIN_SCALE_PID_FILE)
   
    def do_sleep(self):
        time.sleep(180)

    def update_ha_info(self, ns_obj):
        """ Updates HA info if peer changed """
        self.state = ns_obj.state
         
        if self.state in [HaStatus.STANDALONE, HaStatus.PRIMARY, HaStatus.CCO]:
            self.check_rain_scale()

        if self.state in [HaStatus.SECONDARY, HaStatus.NON_CCO]:
            self.signal_rain_scale()

        if ns_obj.peer_ip:
            self.gather_peer_info(ns_obj.peer_ip, self.project)
            if self.inc_mode != ns_obj.inc_mode :
                self.inc_mode = ns_obj.inc_mode
            # Create target instance for secondary node if vips custom metadata is present
            if self.state == HaStatus.SECONDARY:
                self.check_vips_custom_metadata()
            return
        self.peer_ip = None

    def handle_failover(self):
        """ Handles Failover """
        if self.state == HaStatus.CCO:
            logging.info("Cluster CLIPs Movement")
            clips = self.get_all_clips()
            self.migrate_all_clips(clips)
        elif self.peer_ip:
            self.migrate_ip_from_peer()
            self.perform_metadata_sync()

    def get_all_clips(self):
        """ Get all CLIPs from the packet engine """
        clips = []
        if self.nitro is None:
            self.nitro = nitro_cli()
            self.parser = nitro_cli_output_parser()

        nsipdict = self.nitro.get_nsip()

        for ip in nsipdict['nsip']:
            if ip['type'] == 'CLIP':
                clips.append(ip["ipaddress"])

        return clips

    def migrate_all_clips(self, clips):
        logging.info("Moving CLIP IP")
        prev_node = None
        credentials = GoogleCredentials.get_application_default()
        service = discovery.build('compute',
                                  'v1',
                                  credentials=credentials, cache_discovery=False)
        addresses = service.addresses().list(
                        project=self.project,
                        region=self.zone[:self.zone.rfind('-')])
        while addresses is not None:
            try:
                if self.state == HaStatus.PRIMARY:
                    type = "HA"
                elif self.state == HaStatus.CCO:
                    type = "CLIP"
                address = addresses.execute()
                for addr in address["items"]:
                    if addr["address"] not in clips:
                        logging.debug("Skipping address - %s", addr["address"])
                        continue

                    if addr["status"] == "IN_USE":
                        prev_node = addr["users"][0].rsplit('/', 1)[-1]
                    else:
                        prev_node = None
                        logging.info("Skipping address %s as prev node is not present", addr["address"])

                    if self.name == prev_node:
                        logging.info("Skipping address: already moved - %s", addr["address"])
                        continue

                    logging.info("CLIP address %s used by %s", addr["address"], prev_node)
                    if prev_node != None:
                        self.detach_clip_prev_node(service, addr["address"], prev_node, addr["subnetwork"])
                    self.attach_clip_ip(service, addr["address"], addr["subnetwork"])

                addresses = service.addresses().list_next(previous_request=addresses,
                                    previous_response=address)
                log_str = "CLIP is moved successfully"
                out = self.nitro.get_cloud_apistatus("GOOGLECLOUD", type)
                if self.parser.success(out):
                    status_str = out[u'cloudapistatus'][0][u'status']
                    logging.info("Previous status of the CLI is " + status_str)
                    self.nitro.set_cloud_apistatus("GOOGLECLOUD", type, "SUCCESS", log_str)
                    self.nitro.rm_cloud_apistatus("GOOGLECLOUD", type)
                    logging.info("Changing status of the CLI to SUCCESS")
                else:
                    self.nitro.add_cloud_apistatus("GOOGLECLOUD", type, "SUCCESS", log_str)
                    self.nitro.rm_cloud_apistatus("GOOGLECLOUD", type)
                    logging.info("Adding status of the CLI to SUCCESS")
            except Exception as err:
                logging.exception("Encountered error on getting address list while moving"
                                                "CLIP:%s", str(err))
                log_str = "Encountered error on getting address list while moving CLIP:%s" + str(err)
                out = self.nitro.get_cloud_apistatus("GOOGLECLOUD", type)
                if self.parser.success(out):
                    status_str = out[u'cloudapistatus'][0][u'status']
                    logging.info("Previous status of the CLI is " + status_str)
                    self.nitro.set_cloud_apistatus("GOOGLECLOUD", type, "FAILURE", log_str)
                    logging.info("Changing status of the CLI to FAILURE")
                else:
                    self.nitro.add_cloud_apistatus("GOOGLECLOUD", type, "FAILURE", log_str)
                    logging.info("Adding status of the CLI to FAILURE")
                return
        return

    def detach_clip_prev_node(self,service, clip_ip, prev_node, clip_ip_subnetwork):
        try:
            prev_node_interfaces = service.instances().get(
                project=self.project,
                zone=self.zone,
                instance=prev_node).execute()["networkInterfaces"]
        except Exception as err :
            logging.error("Prev Node Interfaces fetch failed. Skipping ip movement")
            logging.error("Exception : %s", str(err))
            return

        for intf in prev_node_interfaces:
            if intf["subnetwork"] == clip_ip_subnetwork:
                aliasIps = intf["aliasIpRanges"]
                for ip in aliasIps:
                    if clip_ip == ip["ipCidrRange"].rsplit('/', 1)[0]:
                        logging.info("Detaching CLIP Alias IP %s", clip_ip)
                        aliasIps.remove(ip)

                        request_body = {"aliasIpRanges" : aliasIps,
                            "fingerprint" : intf["fingerprint"]}
                        request = service.instances().updateNetworkInterface(
                            project=self.project,
                            zone=self.zone,
                            instance=prev_node,
                            networkInterface=intf["name"],
                            body=request_body)
                        try:
                            response = request.execute()
                            result = self.wait_for_response(service,
                                            self.project,
                                            self.zone,
                                            response["name"])
                            if result == False:
                                logging.error("Error while removing the alias ip CLIP : %s.",
                                            clip_ip)
                        except Exception as err:
                            logging.exception("Encountered error while removing "
                                                  "alias ip: %s", str(err))
                            return
                        break
                break
        return

    def attach_clip_ip(self, service, clip_ip, clip_ip_subnetwork):
        try:
            self_interfaces = service.instances().get(
                project=self.project,
                zone=self.zone,
                instance=self.name).execute()["networkInterfaces"]
        except Exception as err :
            logging.error("Self Interfaces fetch failed. Skipping ip movement")
            logging.error("Exception : %s", str(err))
            return

        for intf in self_interfaces:
            if intf["subnetwork"] == clip_ip_subnetwork:
                if "aliasIpRanges" in intf:
                    aliasIps = intf["aliasIpRanges"]
                else:
                    aliasIps = []
                clip_list = {u"ipCidrRange": clip_ip + "/32"}

                logging.info("Attaching CLIP IP : %s", clip_ip)
                aliasIps.append(clip_list)
                request_body = {"aliasIpRanges" : aliasIps,
                    "fingerprint" : intf["fingerprint"]}
                request = service.instances().updateNetworkInterface(
                    project=self.project,
                    zone=self.zone,
                    instance=self.name,
                    networkInterface=intf["name"],
                    body=request_body)
                try:
                    response = request.execute()
                    result = self.wait_for_response(service,
                                self.project,
                                self.zone,
                                response["name"])
                    if result == False:
                        logging.exception("Error while attaching the alias ip CLIP : %s.",
                                            clip_ip)
                except Exception as err:
                    logging.error("Encountered error while attaching "
                                            "alias ip: %s", str(err))
                    return
                break
        return

    def is_external_ip_present(self, state):
        """Check if EIP is present in primary node"""
        logging.info("Check EIP")
        credentials = GoogleCredentials.get_application_default()
        service = discovery.build('compute',
                                   'v1',
                                    credentials=credentials, cache_discovery=False)
        if state == HaStatus.PRIMARY:
            try:
                interface = service.instances().get(
                          project=self.project,
                          zone=self.zone,
                         instance=self.name).execute()["networkInterfaces"]
            except Exception as err:
                logging.exception("Some error occured- %s", err)
                return False
        else:
            try:
                interface = service.instances().get(
                            project=self.peer_project,
                            zone=self.peer_zone,
                            instance=self.peer_name).execute()["networkInterfaces"]
            except Exception as err:
                logging.exception("Peer resource not found- %s", err)
                return False
        vips = self.get_vips()
        interface_count = len(interface)
        for i in range(0, interface_count):
            if i == self.management_nic:
                continue
            if "accessConfigs" in  interface[i] \
                and len(interface[i]["accessConfigs"]) > 0 and interface[i]["networkIP"] in vips :
                for config in interface[i]["accessConfigs"]:
                    if "natIP" in config:
                        return True
        return False

    def alias_ip_present(self, state):
        "Check if alias IP configured as VIP on primary node"
        logging.info("Check alias ip")
        credentials = GoogleCredentials.get_application_default()
        service = discovery.build('compute',
                                   'v1',
                                    credentials=credentials, cache_discovery=False)
        if state == HaStatus.PRIMARY:
            instance_name = self.name
            zone = self.zone
            project = self.project
        else:
            instance_name = self.peer_name
            zone = self.peer_zone
            project = self.peer_project
        vips = self.get_vips()
        try:
            interface = service.instances().get(
                project=project,
                zone=zone,
                instance=instance_name).execute()["networkInterfaces"]
            for i in range(0, len(interface)):
                if "aliasIpRanges" in interface[i]:
                    if self.inc_mode == "DISABLED":
                    #If INC mode is Disabled, customer is using old config
                        return True
                    for vip in vips:
                        for alias_ip in interface[i]["aliasIpRanges"]:
                            if IPAddress(str(vip)) in IPNetwork(str(alias_ip["ipCidrRange"])):
                                return True
        except Exception as err:
            logging.exception("Could not get alias ip info: %s", err)
        return False

    def get_missing_instance_iam(self, state):
        """Check for missing IAM permissions based on EIP,
        alias IPs and forwarding rules"""
        ha_configuration = "HA_WITH"
        no_eip_pip = True
        if self.is_external_ip_present(state):
            logging.debug("Found EIP")
            ha_configuration = ha_configuration + "_EIP"
            no_eip_pip = False
        if self.alias_ip_present(state):
            ha_configuration = ha_configuration + "_PIP"
            no_eip_pip  = False
        if no_eip_pip is True:
            ha_configuration = "HA_NOT_CONFIGURED"
        if self.state in [HaStatus.CCO, HaStatus.NON_CCO]:
            ha.configuration = "HA_CLUSTER_CLIP_WITH_PIP"
        logging.info("HA configuration-%s", ha_configuration)
        missing_instance_iam = self.check_iam(ha_configuration)
        return missing_instance_iam

    def do_periodic(self):
        """Do periodic cloud specific tasks"""
        if self.peer_ip:
            missing_instance_iam = self.get_missing_instance_iam(self.state)
            if missing_instance_iam:
                with open(IAM_NOT_OK_FILE,'w') as iam_file :
                    iam_file.write(json.dumps({"instance_iam":list(missing_instance_iam)}))
                return
        if os.path.exists(IAM_NOT_OK_FILE):
            os.remove(IAM_NOT_OK_FILE)
