#!/usr/bin/python
###############################################################################
#
#  Copyright (c) 2019-2023 Citrix Systems, Inc.
#  All rights reserved.
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions are met:
#      * Redistributions of source code must retain the above copyright
#        notice, this list of conditions and the following disclaimer.
#      * Redistributions in binary form must reproduce the above copyright
#        notice, this list of conditions and the following disclaimer in the
#        documentation and/or other materials provided with the distribution.
#      * Neither the name of the Citrix Systems, Inc. nor the
#        names of its contributors may be used to endorse or promote products
#        derived from this software without specific prior written permission.
#
#  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
#  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
#  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
#  ARE DISCLAIMED. IN NO EVENT SHALL CITRIX SYSTEMS, INC. BE LIABLE FOR ANY
#  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
#  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
#  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
#  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
#  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
#  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
###############################################################################
""" Generic HA daemon """
import logging

class CloudPlatform() :
    """ Class to store the platform IDs """
    AWS = 1
    AZURE = 3
    GCP = 4

class Cloud():
    """ This class contains cloud related variables and methods """
    def __init__(self, ns_obj):
        if  ns_obj.cloudplatform == CloudPlatform.GCP:
            from src.gcp.ns_vpx_gcp_utils import GcpInstanceInfo
            try:
                self.cloudhandler = GcpInstanceInfo(ns_obj)
            except Exception as exception:
                logging.error("Error while creating cloud Instance, %s" \
                , exception)
        elif ns_obj.cloudplatform == CloudPlatform.AWS:
            from src.aws.ns_vpx_aws_utils import AwsInstanceInfo
            try:
                self.cloudhandler = AwsInstanceInfo(ns_obj)
            except Exception as exception:
                logging.error("Error while creating cloud Instance, %s" \
                , exception)
        elif ns_obj.cloudplatform == CloudPlatform.AZURE:
            from src.azure.ns_vpx_azure_utils import AzureInstanceInfo
            try:
                self.cloudhandler = AzureInstanceInfo(ns_obj)
            except Exception as exception:
                logging.error("Error while creating cloud Instance, %s" \
                , exception)
        else:
            logging.error("HA Daemon is not supported in this platform  %s" \
            , ns_obj.cloudplatform)
            raise Exception("HA Daemon is not supported in this platform  %s" \
            , ns_obj.cloudplatform)

    def handle_failover(self):
        """ Executes failover operations """
        try:
            self.cloudhandler.handle_failover()
        except Exception as exception:
            logging.error("Error while triggering cloud failover, %s" \
            , exception)

    def do_periodic(self):
        """ Periodic checks are done here """
        try:
            self.cloudhandler.do_periodic()
        except Exception as exception:
            logging.error("Error while triggering IAM periodic check, %s" \
            , exception)
            logging.info("Please check Internet connectivity...")

    def do_sleep(self):
       """ Cloud specific sleep """
       try:
           self.cloudhandler.do_sleep()
       except Exception as exception:
           logging.error("Error while calling do_sleep, %s " \
           , exception)
           logging.info("Error while calling do_sleep...") 

    def update_ha_info(self, ns_obj):
        """ Update the ha state and peer ip if needed """
        self.cloudhandler.update_ha_info(ns_obj)
