#!/usr/bin/python
###############################################################################
#
#  Copyright (c) 2022-2024 Citrix Systems, Inc.
#  All rights reserved.
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions are met:
#      * Redistributions of source code must retain the above copyright
#        notice, this list of conditions and the following disclaimer.
#      * Redistributions in binary form must reproduce the above copyright
#        notice, this list of conditions and the following disclaimer in the
#        documentation and/or other materials provided with the distribution.
#      * Neither the name of the Citrix Systems, Inc. nor the
#        names of its contributors may be used to endorse or promote products
#        derived from this software without specific prior written permission.
#
#  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
#  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
#  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
#  ARE DISCLAIMED. IN NO EVENT SHALL CITRIX SYSTEMS, INC. BE LIABLE FOR ANY
#  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
#  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
#  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
#  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
#  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
#  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
###############################################################################
""" Generic HA daemon """
import logging
import logging.handlers
import time
import os
import sys
import psutil
import signal
from src.ns_cloud import Cloud
from src.ns_ha import NS
from src.ha_state import HaStatus

_LOG_FILE_NAME = '/var/log/cloud-ha-daemon.log'
ENABLE_DEBUG_FILE='/nsconfig/.cloudlog'
MaxBytes = 5*1024
LogBackupCount = 20

class HaDaemon():
    """
    This is a cloud agnostic daemon class for HA failover
    """
    def __init__(self):
        logging.info("Starting daemon")
        self.ns_obj = NS()
        self.cloud = Cloud(self.ns_obj)
        self.handle_failover()
        signal.signal(signal.SIGUSR2, self.handle_signal)

    def handle_signal(self, signum, frame) :
        """ Signal handler for any state change from Primary to Sec """
        logging.info("Caught signal %d", signum)
        if signum == signal.SIGUSR2:
            logging.info("HA State change is detected")
            self.handle_failover()

    def update_ha_info(self):
        """ Updates ha state and peer ip """
        self.ns_obj.update_ha_info()
        self.cloud.update_ha_info(self.ns_obj)
        logging.info("My HA Status is %s", self.ns_obj.state)

    def handle_failover(self):
        """ Handles failover"""
        self.update_ha_info()
        if self.ns_obj.state in [HaStatus.PRIMARY, HaStatus.CCO]:
            self.cloud.handle_failover()

    def do_periodic(self):
        """ Do periodic update and checks """
        #-- Do periodic tasks for the cloud
        self.cloud.do_periodic()
        self.cloud.do_sleep()
        #-- Update the current HA Status
        self.update_ha_info()

def is_daemon_running():
    """Exit if daemon is already running"""
    daemon_pid_file = '/var/run/nscloudhaagent.pid'
    if os.path.exists(daemon_pid_file) and \
		os.stat(daemon_pid_file).st_size != 0:
        with open(daemon_pid_file,'r') as pid:
            daemon_pid = pid.read()
            if psutil.pid_exists(int(daemon_pid)):
                logging.debug("Daemon is already running")
                sys.exit(0)
    with open(daemon_pid_file,'w') as pid :
        pid.write(str(os.getpid()))

def Rotate_Log_Periodically():
    #-- Initialize logging config and advance debug if .debug file is present
    logger = logging.getLogger()
    handler = logging.handlers.RotatingFileHandler(filename=_LOG_FILE_NAME, maxBytes=MaxBytes, backupCount=LogBackupCount, delay=True)
    if os.path.exists(ENABLE_DEBUG_FILE):
        logging.basicConfig(format='%(asctime)s \
	%(levelname)s %(message)s',datefmt='%Y-%m-%d %H:%M:%S',filename=_LOG_FILE_NAME, level=logging.DEBUG)
        handler.setLevel(logging.DEBUG)
    else:
        logging.basicConfig(format='%(asctime)s \
	%(levelname)s %(message)s',datefmt='%Y-%m-%d %H:%M:%S',filename=_LOG_FILE_NAME, level=logging.INFO)
        handler.setLevel(logging.INFO)
    formatter = logging.Formatter('%(asctime)s: %(levelname)-2s: %(message)s',datefmt='%Y-%m-%d %H:%M:%S')
    handler.setFormatter(formatter)
    logger.handlers=[handler]

def main():
    """
    Initialize HA Daemon
    Initialize logging config

    """
    #-- Initialize logging config and advance debug if .debug file is present
    Rotate_Log_Periodically()
    #Check if daemon is already running
    is_daemon_running()
    #-- Instantiate a HaDaemon Object
    ha = HaDaemon()
    while True:
        ha.do_periodic()

if __name__ == '__main__':
    main()
