# Copyright (C) 2018-2023. Cloud Software Group, Inc. All Rights Reserved. Confidential & Proprietary.

"""RSA key encoding/decoding.

Encode keys using the encode_public/encode_private functions. They will return
a base64-encoded representation based on the .NET RSA Key XML Format.

Decode keys using the decode_public/decode_private functions. They will return
the original rsa.PublicKey/rsa.PrivateKey objects.
"""
from ccauth.util import asstring, asbytes
from ccauth.error import ensure_success_code
from ctypes import *
import ccauth.interop as interop
import rsa


def encode_publickey(pub_key):
    """Encodes an rsa.PublicKey.

    :param pub_key: the public key to encode, see py:class:`rsa.PublicKey`
    :returns:       a base64-encoded .NET RSA Key (XML Format) public key
    """
    rsaparams = interop.CCAuthRsaParameters()
    rsaparams.modulus = interop.CCAuthRsaParam._marshal(pub_key.n)
    rsaparams.exponent = interop.CCAuthRsaParam._marshal(pub_key.e, False)
    
    encodedkey_ptr = c_char_p()
    
    err = interop.ccauth_encode_rsakey(byref(rsaparams), byref(encodedkey_ptr))
    ensure_success_code(err, 'could not encode public key.')

    try:
        return asstring(string_at(encodedkey_ptr))
    finally:
        interop.ccauth_free(encodedkey_ptr)


def decode_publickey(encoded_key):
    """Decodes a base64-encoded .NET RSA Key (XML Format) public key

    :param encoded_key: the encoded key to decode
    :returns:           an instance of py:class:`rsa.PublicKey`
    """
    rsaparams = _decode_rsakey(encoded_key)
    n = rsaparams.modulus._unmarshal()
    e = rsaparams.exponent._unmarshal()
    return rsa.PublicKey(n, e)


def encode_privatekey(priv_key):
    """Encodes an rsa.PrivateKey.

    :param priv_key:    the private key to encode, see py:class:`rsa.PrivateKey`
    :returns:           a base64-encoded .NET RSA Key (XML Format) private key
    """
    rsaparams = interop.CCAuthRsaParameters()
    rsaparams.modulus = interop.CCAuthRsaParam._marshal(priv_key.n)
    rsaparams.exponent = interop.CCAuthRsaParam._marshal(priv_key.e, False)
    rsaparams.p = interop.CCAuthRsaParam._marshal(priv_key.p)
    rsaparams.q = interop.CCAuthRsaParam._marshal(priv_key.q)
    rsaparams.dp = interop.CCAuthRsaParam._marshal(priv_key.exp1)
    rsaparams.dq = interop.CCAuthRsaParam._marshal(priv_key.exp2)
    rsaparams.inverse_q = interop.CCAuthRsaParam._marshal(priv_key.coef)
    rsaparams.d = interop.CCAuthRsaParam._marshal(priv_key.d)
    
    encodedkey_ptr = c_char_p()
    
    err = interop.ccauth_encode_rsakey(byref(rsaparams), byref(encodedkey_ptr))
    ensure_success_code(err, 'could not encode private key.')

    try:
        return asstring(string_at(encodedkey_ptr))
    finally:
        interop.ccauth_free(encodedkey_ptr)


def decode_privatekey(encoded_key):
    """Decodes a base64-encoded .NET RSA Key (XML Format) private key

    :param encoded_key: the encoded key to decode
    :returns:           an instance of py:class:`rsa.PrivateKey`
    """
    rsaparams = _decode_rsakey(encoded_key)
    n = rsaparams.modulus._unmarshal()
    e = rsaparams.exponent._unmarshal()
    d = rsaparams.d._unmarshal()
    p = rsaparams.p._unmarshal()
    q = rsaparams.q._unmarshal()
    return rsa.PrivateKey(n, e, d, p, q)


def _decode_rsakey(encoded_key):
    ptr = c_void_p()
    err = interop.ccauth_decode_rsakey(asbytes(encoded_key), byref(ptr))
    ensure_success_code(err, 'could not decode rsa key.')
    return cast(ptr, POINTER(interop.CCAuthRsaParameters))[0]
