# Copyright (C) 2018-2023. Cloud Software Group, Inc. All Rights Reserved. Confidential & Proprietary.

"""HTTP request definitions."""
from ccauth.util import AttributedObject, asstring
import ccauth.interop as interop
from ctypes import *


class RequestMethod:
    """HTTP request methods."""
    GET = 0
    POST = 1
    PUT = 2
    DELETE = 3
    OPTIONS = 4
    HEAD = 5
    TRACE = 6
    PATCH = 7

    @staticmethod
    def tostring(method):
        """Converts an HTTP method to its string representation.

        :param method:  the HTTP method to convert
        :returs:        the str representation matching the given HTTP method
        """
        if method == RequestMethod.GET: return 'GET'
        if method == RequestMethod.POST: return 'POST'
        if method == RequestMethod.PUT: return 'PUT'
        if method == RequestMethod.DELETE: return 'DELETE'
        if method == RequestMethod.OPTIONS: return 'OPTIONS'
        if method == RequestMethod.HEAD: return 'HEAD'
        if method == RequestMethod.TRACE: return 'TRACE'
        if method == RequestMethod.PATCH: return 'PATCH'

        raise ValueError('method')

    @staticmethod
    def fromstring(method):
        """Converts to an HTTP method from its string representation.

        :param method:  the HTTP method to convert
        :returs:        the HTTP method matching the given str representation
        """
        method = method.upper()

        if method == 'GET': return RequestMethod.GET
        if method == 'POST': return RequestMethod.POST
        if method == 'PUT': return RequestMethod.PUT
        if method == 'DELETE': return RequestMethod.DELETE
        if method == 'OPTIONS': return RequestMethod.OPTIONS
        if method == 'HEAD': return RequestMethod.HEAD
        if method == 'TRACE': return RequestMethod.TRACE
        if method == 'PATCH': return RequestMethod.PATCH

        raise ValueError('method')


class RequestInfo(AttributedObject):
    """HTTP request info."""
    _attrs_ = [
        ('method', int),
        ('uri', str)
    ]

    def __init__(self, **kwargs):
        self.method = None
        self.uri = None

        AttributedObject.__init__(self, **kwargs)


class RequestMessage(AttributedObject):
    """HTTP request message."""
    _attrs_ = [
        ('method', int),
        ('uri', str),
        ('headers', dict),
        ('content', bytes)
    ]

    def __init__(self, **kwargs):
        self.method = None
        self.uri = None
        self.headers = None
        self.content = None

        AttributedObject.__init__(self, **kwargs)

    @staticmethod
    def _unmarshal(ptr):
        native = cast(ptr, POINTER(interop.CCAuthRequest))[0];
        headers = {}
        content = None

        if native.content_size > 0:
            content = bytes(POINTER(c_ubyte * native.content_size).from_buffer(native.content)[0])

        for i in range(0, native.header_count):
            h = native.headers[i]
            name = asstring(string_at(h.name))
            value = asstring(string_at(h.value))
            headers[name] = value

        return RequestMessage(
            method = native.method,
            uri = asstring(string_at(native.url)),
            headers = headers,
            content = bytes(content) if content is not None else None
        )


class ResponseMessage(AttributedObject):
    """HTTP response message."""
    _attrs_ = [
        ('status', int),
        ('content', bytes)
    ]

    def __init__(self, **kwargs):
        self.status = None
        self.content = None

        AttributedObject.__init__(self, **kwargs)
