# Copyright (C) 2019-2023. Cloud Software Group, Inc. All Rights Reserved. Confidential & Proprietary.

"""Access control definitions."""
from ccauth.security.identity import AdminClaims
from ccauth.util import AttributedObject, isempty, asbytes
import ccauth.interop as interop
from ccauth.error import ensure_success_code
from ctypes import *


class AccessPolicy(AttributedObject):
    """Access policy object"""
    _attrs_ = [
        ('permissions', [list, tuple]),
        ('scopes', [list, tuple])
    ]

    def __init__(self, **kwargs):
        self.permissions = None
        self.scopes = None

        AttributedObject.__init__(self, **kwargs)

    @staticmethod
    def _unmarshal(ptr):
        native = cast(ptr, POINTER(interop.CCAuthAccessPolicy))[0]

        return AccessPolicy(
            permissions=interop.CCAuthArray._unmarshal(native.permissions) if native.permissions is not None else None,
            scopes=interop.CCAuthArray._unmarshal(native.scopes) if native.scopes is not None else None
        )


class AccessQueryContext(AttributedObject):
    """Access query context"""
    _attrs_ = [
        ('customer_id', str),
        ('user_id', str),
        ('service_name', str),
        ('service_instance', str),
        ('admin_claims', AdminClaims),
        ('rbac_version', int),
        ('bearer_token', str)
    ]

    def __init__(self, **kwargs):
        self.customer_id = None
        self.user_id = None
        self.service_name = None
        self.service_instance = None
        self.admin_claims = None
        self.rbac_version = None
        self.bearer_token = None

        AttributedObject.__init__(self, **kwargs)


def build_required_policy(template, customer_id, service_name=None, service_profile=None):
    """Builds a required access policy.

    :param template:            the template py:class:`ccauth.security.AccessPolicy`
    :param customer_id:         the target customer id
    :param service_name:        the target service name
    :param service_profile:     the target service profile
    :returns:                   an instance of py:class:`ccauth.security.AccessPolicy`
    """
    if template is None or not isinstance(template, AccessPolicy):
        raise ValueError("'template' must be of type 'AccessPolicy'.")

    if isempty(customer_id) and isempty(service_name) and isempty(service_profile):
        raise ValueError("'customer_id', 'service_name', and 'service_profile' cannot all be empty.")

    policy = interop.CCAuthAccessPolicy._marshal(template)
    policy_ptr = c_void_p()

    err = interop.ccauth_required_policy(
        byref(policy),
        asbytes(customer_id) if customer_id is not None else None,
        asbytes(service_name) if service_name is not None else None,
        asbytes(service_profile) if service_profile is not None else None,
        byref(policy_ptr))

    ensure_success_code(err, 'could not build required policy.')

    try:
        return AccessPolicy._unmarshal(policy_ptr)
    finally:
        interop.ccauth_free_policy(policy_ptr)
