# Copyright (C) 2018-2023. Cloud Software Group, Inc. All Rights Reserved. Confidential & Proprietary.

"""Identity validation definitions."""
from ccauth.util import AttributedObject, asstring
import ccauth.interop as interop
from ctypes import *
from .rbac import *


class UserMode:
    """User modes as defined in libccauth."""
    DEFAULT = 0
    ADMIN = 1
    END_USER = 2
    EITHER = 3

    @staticmethod
    def tostring(mode):
        """Converts a user mode to its string representation.

        :param mode:    the mode to convert
        :returns:        the str representation matching the given mode
        """
        if mode == UserMode.DEFAULT:
            return 'DEFAULT'
        if mode == UserMode.ADMIN:
            return 'ADMIN'
        if mode == UserMode.END_USER:
            return 'END_USER'
        if mode == UserMode.EITHER:
            return 'EITHER'

        raise ValueError('mode')


class AuthorizationCode:
    """Authorization codes as defined in libccauth."""
    OK = 0
    SERVICE_UNAVAILABLE = 1
    INVALID_SERVICE_KEY = 2
    INVALID_BEARER_TOKEN = 3
    INVALID_CUSTOMER = 4
    INVALID_USER = 5

    @staticmethod
    def tostring(code):
        """Converts an authorization code to its string representation.

        :param code:    the code to convert
        :returns:        the str representation matching the given code
        """
        if code == AuthorizationCode.OK: return 'OK'
        if code == AuthorizationCode.SERVICE_UNAVAILABLE: return 'SERVICE_UNAVAILABLE'
        if code == AuthorizationCode.INVALID_SERVICE_KEY: return 'INVALID_SERVICE_KEY'
        if code == AuthorizationCode.INVALID_BEARER_TOKEN: return 'INVALID_BEARER_TOKEN'
        if code == AuthorizationCode.INVALID_CUSTOMER: return 'INVALID_CUSTOMER'
        if code == AuthorizationCode.INVALID_USER: return 'INVALID_USER'

        raise ValueError('code')


class AdministratorType:
    """Administrator types as defined in libccauth."""
    NONE = 0
    FULL = 1
    CUSTOM = 2

    @staticmethod
    def tostring(mode):
        """Converts an administrator type to its string representation.

        :param mode:    the type to convert
        :returns:        the str representation matching the given type
        """
        if mode == AdministratorType.NONE:
            return 'NONE'
        if mode == AdministratorType.FULL:
            return 'FULL'
        if mode == AdministratorType.CUSTOM:
            return 'CUSTOM'

        raise ValueError('mode')


class UserClaims(AttributedObject):
    """User (universal) claims object"""
    _attrs_ = [
        ('oid', str),
        ('sid', str),
        ('upn', str),
        ('email', str),
        ('customers', [list, tuple])
    ]

    def __init__(self, **kwargs):
        self.oid = None
        self.sid = None
        self.upn = None
        self.email = None
        self.customers = None

        AttributedObject.__init__(self, **kwargs)

    @staticmethod
    def _unmarshal(ptr):
        native = cast(ptr, POINTER(interop.CCAuthUserClaims))[0]

        return UserClaims(
            oid=asstring(string_at(native.oid)) if native.oid is not None else None,
            sid=asstring(string_at(native.sid)) if native.sid is not None else None,
            upn=asstring(string_at(native.upn)) if native.upn is not None else None,
            email=asstring(string_at(native.email)) if native.email is not None else None,
            customers=interop.CCAuthArray._unmarshal(native.customers) if native.customers is not None else None
        )


class DirectoryClaims(AttributedObject):
    """Directory (universal) claims object"""
    _attrs_ = [
        ('forest', str),
        ('domain', str),
        ('identityprovider', str),
        ('tenant', str)
    ]

    def __init__(self, **kwargs):
        self.forest = None
        self.domain = None
        self.identityprovider = None
        self.tenant = None

        AttributedObject.__init__(self, **kwargs)

    @staticmethod
    def _unmarshal(ptr):
        native = cast(ptr, POINTER(interop.CCAuthDirectoryClaims))[0]

        return DirectoryClaims(
            forest=asstring(string_at(native.forest)) if native.forest is not None else None,
            domain=asstring(string_at(native.domain)) if native.domain is not None else None,
            identityprovider=asstring(string_at(native.identityprovider)) if native.identityprovider is not None else None,
            tenant=asstring(string_at(native.tenant)) if native.tenant is not None else None
        )


class AdminClaims(AttributedObject):
    """Admin (universal) claims object"""
    _attrs_ = [
        ('groups', [list, tuple])
    ]

    def __init__(self, **kwargs):
        self.groups = None

        AttributedObject.__init__(self, **kwargs)

    @staticmethod
    def _unmarshal(ptr):
        native = cast(ptr, POINTER(interop.CCAuthAdminClaims))[0]

        return AdminClaims(
            groups= interop.CCAuthArray._unmarshal(native.groups) if native.groups is not None else None
        )


class IdentityValidationContext(AttributedObject):
    """Identity validation context"""
    _attrs_ = [
        ('customer_id', str),
        ('allow_custom_admins', bool),
        ('user_mode', int),
        ('fetch_user_claims', bool),
        ('fetch_user_groups', bool),
        ('skip_service_profile', bool),
        ('caller_service_name', str),
        ('caller_service_instance', str),
        ('fetch_directory_claims', bool),
        ('fetch_admin_claims', bool),
        ('rbac_version', int)
    ]

    def __init__(self, **kwargs):
        self.customer_id = None
        self.allow_custom_admins = None
        self.user_mode = UserMode.DEFAULT
        self.fetch_user_claims = None
        self.fetch_user_groups = None
        self.skip_service_profile = None
        self.caller_service_name = None
        self.caller_service_instance = None
        self.fetch_directory_claims = None
        self.fetch_admin_claims = None
        self.rbac_version = RbacVersion.DEFAULT

        AttributedObject.__init__(self, **kwargs)


class IdentityValidationResult(AttributedObject):
    """Identity validation result"""
    _attrs_ = [
        ('code', int),
        ('customer_id', str),
        ('user_id', str),
        ('user_principal', str),
        ('user_name', str),
        ('sub', str),
        ('amr', [list, tuple]),
        ('admin_type', int),
        ('user_claims', UserClaims),
        ('directory_claims', DirectoryClaims),
        ('user_groups', [list, tuple]),
        ('admin_claims', AdminClaims),
        ('service_name', str),
        ('service_instance', str),
        ('service_profile', str),
        ('service_multitenant', bool),
        ('bearer_token', str),
        ('service_key', str)
    ]

    def __init__(self, **kwargs):
        self.code = None
        self.customer_id = None
        self.user_id = None
        self.user_principal = None
        self.user_name = None
        self.sub = None
        self.amr = None
        self.admin_type = AdministratorType.NONE
        self.user_claims = None
        self.directory_claims = None
        self.user_groups = None
        self.admin_claims = None
        self.service_name = None
        self.service_instance = None
        self.service_profile = None
        self.service_multitenant = None
        self.bearer_token = None
        self.service_key = None

        AttributedObject.__init__(self, **kwargs)

    @staticmethod
    def _unmarshal(ptr):
        native = cast(ptr, POINTER(interop.CCAuthIdentity))[0]

        return IdentityValidationResult(
            code=native.code,
            customer_id=asstring(string_at(native.customer)) if native.customer is not None else None,
            user_id=asstring(string_at(native.user_id)) if native.user_id is not None else None,
            user_principal=asstring(string_at(native.user_principal)) if native.user_principal is not None else None,
            user_name=asstring(string_at(native.user_name)) if native.user_name is not None else None,
            sub=asstring(string_at(native.sub)) if native.sub is not None else None,
            amr=interop.CCAuthArray._unmarshal(native.amr) if native.amr is not None else None,
            admin_type=native.admin_type,
            user_claims=UserClaims._unmarshal(native.user_claims) if native.user_claims is not None else None,
            directory_claims=DirectoryClaims._unmarshal(native.directory_claims) if native.directory_claims is not None else None,
            user_groups=interop.CCAuthArray._unmarshal(native.user_groups) if native.user_groups is not None else None,
            admin_claims=AdminClaims._unmarshal(native.admin_claims) if native.admin_claims is not None else None,
            service_name=asstring(string_at(native.service_name)) if native.service_name is not None else None,
            service_instance=asstring(string_at(native.service_instance)) if native.service_instance is not None else None,
            service_profile=asstring(string_at(native.service_profile)) if native.service_profile is not None else None,
            service_multitenant=native.service_multitenant == 1,
            bearer_token=asstring(string_at(native.bearer_token)) if native.bearer_token is not None else None,
            service_key=asstring(string_at(native.service_key)) if native.service_key is not None else None
        )
