# Copyright (C) 2018-2023. Cloud Software Group, Inc. All Rights Reserved. Confidential & Proprietary.

"""Service key definitions"""
from ccauth.util import AttributedObject, isempty, asstring, asbytes
from ccauth.error import ensure_success_code
import ccauth.interop as interop
from ctypes import *


class SigningAlgorithm:
    """Signing algorithms as defined in libccauth."""
    DEFAULT = 0

    RSA_SHA256_V1 = 1
    RSA_SHA256_V2 = 2

    MIN_VERSION = RSA_SHA256_V1
    MIN_SUPPORTED_VERSION = RSA_SHA256_V1
    MAX_VERSION = RSA_SHA256_V2
    CURRENT_VERSION = RSA_SHA256_V2

    @staticmethod
    def tostring(algorithm):
        """Converts a signing algorithm to its string representation.

        :param algorithm:   the signing algorithm to convert
        :returns:           the str representation matching the given signing algorithm
        """
        if algorithm == SigningAlgorithm.RSA_SHA256_V1: return 'CC-RSA-SHA256'
        if algorithm == SigningAlgorithm.RSA_SHA256_V2: return 'CC-RSA-SHA256-2'

        raise ValueError('algorithm')

    @staticmethod
    def fromstring(algorithm):
        """Converts to a signing algorithm from its string representation.

        :param algorithm:   the signing algorithm to parse
        :returs:            the signing algorithm matching the given str representation
        """
        algorithm = algorithm.upper()

        if algorithm == 'CC-RSA-SHA256': return SigningAlgorithm.RSA_SHA256_V1
        if algorithm == 'CC-RSA-SHA256-2': return SigningAlgorithm.RSA_SHA256_V2

        raise ValueError('method')

    @staticmethod
    def isvalid(algorithm):
        """Checks whether a given algorithm is valid.

        :param algorithm:   the signing algorithm to validate
        :returns:           whether the algorithm is valid
        """
        return SigningAlgorithm.DEFAULT <= algorithm <= SigningAlgorithm.MAX_VERSION


class ServiceKey(AttributedObject):
    """Service key object"""
    _attrs_ = [
        ('algorithm', int),
        ('service_name', str),
        ('service_instance', str),
        ('signature', str),
        ('timestamp', int),
        ('nonce', str)
    ]

    def __init__(self, **kwargs):
        self.algorithm = None
        self.service_name = None
        self.service_instance = None
        self.signature = None
        self.timestamp = None
        self.nonce = None

        AttributedObject.__init__(self, **kwargs)

    @staticmethod
    def _unmarshal(ptr):
        native = cast(ptr, POINTER(interop.CCAuthServiceKey))[0]

        return ServiceKey(
            algorithm = native.algorithm,
            service_name = asstring(string_at(native.service_name)),
            service_instance = asstring(string_at(native.service_instance)) if native.service_instance is not None else None,
            signature = asstring(string_at(native.signature)),
            nonce = asstring(string_at(native.nonce)) if native.nonce is not None else None,
            timestamp = native.timestamp
        )


def decode_servicekey(servicekey):
    """Decodes a given service key.

    :param servicekey:  the encoded service key to decode
    :returns:           an instance of py:class:`ccauth.security.ServiceKey`
    """
    if isempty(servicekey):
        raise ValueError("'servicekey' is empty.")

    svckey_ptr = c_void_p()
    err = interop.ccauth_decode_servicekey(
        asbytes(servicekey),
        byref(svckey_ptr))

    ensure_success_code(err, 'could not decode service key.')

    try:
        return ServiceKey._unmarshal(svckey_ptr)
    finally:
        interop.ccauth_free_servicekey(svckey_ptr)
