# Copyright (C) 2018-2023. Cloud Software Group, Inc. All Rights Reserved. Confidential & Proprietary.

"""Helper class to add checked/typed access to class attributes."""
from .encoding import asstring, isunicode


class AttributedObject(object):
    """Attributed object"""

    def __init__(self, **kwargs):
        if kwargs is not None:
            for key, value in kwargs.items():
                setattr(self, key, value)

    def __setattr__(self, name, value):
        attr = None

        for entry in self._attrs_:
            if entry[0] == name:
                attr = entry
                break

        if attr is None:
            raise AttributeError("unknown attribute '{0}'".format(name))

        type_list = attr[1]

        if not isinstance(type_list, list):
            type_list = [type_list]

        for attr_type in type_list:
            succeeded, v = AttributedObject._checkattr(value, attr_type)

            if succeeded:
                value = v
                break

        if not succeeded:
            raise AttributeError("unexpected value of type '{0}' instead of '{1}'".format(type(value), type_list))

        object.__setattr__(self, name, value)

    @staticmethod
    def _checkattr(value, attr_type):
        # check attribute type and attempt some basic conversions
        if value is not None and not isinstance(value, attr_type):
            if attr_type == str and (isinstance(value, bytes) or isunicode(value)):
                value = asstring(value)
            elif attr_type == bool and isinstance(value, int):
                value = False if value == 0 else True
            elif attr_type == float and isinstance(value, int):
                value = float(value)
            elif attr_type == int and isinstance(value, float) and int(value) == value:
                value = int(value)
            else:
                return False, None

        return True, value
