# Copyright Citrix Systems, Inc. All rights reserved.

"""CWC Authorization Header.

Use :py:class:`cwc.auth.header.CwcAuthorizationHeader` to represent the custom CWC authoization headers.
"""
from cwc.util import isempty
import re

class CwcAuthorizationHeader:
    """Represents a CWC authorization header."""

    def __init__(self, bearer, servicekey):
        """Initializes a new instance.

        :param bearer:      the bearer token to use
        :param servicekey:  the service key to use
        :returns:           a new instance of :py:class:`cwc.auth.header.CwcAuthorizationHeader`
        """
        if isempty(bearer) and isempty(servicekey):
            raise ValueError()

        self.bearer = bearer
        self.servicekey = servicekey

    def __repr__(self):
        """Returns the string representation for the header.

        :returns:   the string representing the header
        """
        fields = ', '.join(['%s="%s"' % (n, v) for (n, v) in [('bearer', self.bearer), ('service', self.servicekey)] if not isempty(v)])
        return 'CWSAuth %s' % fields

    @classmethod
    def fromstring(cls, header):
        """Parses a CWC authorization header from a string.

        :param header:  the header to parse
        :returns:       a parsed :py:class:`cwc.auth.header.CwcAuthorizationHeader`
        """
        if isempty(header):
            return None

        schema = 'CWSAuth '
        header = header.strip()

        if not re.match(schema, header, re.I):
            raise ValueError('header')

        attrs = header[len(schema):].split(',')
        bearer = None
        service = None

        for pair in attrs:
            fields = pair.split('=', 1)

            name = fields[0].strip()
            value = fields[1].strip().strip('"').strip("'") if len(fields) > 1 else ''

            if re.match(name, 'bearer', re.I):
                bearer = value
            elif re.match(name, 'service', re.I):
                service = value
            else:
                raise ValueError('header')

        return CwcAuthorizationHeader(bearer, service)
