# Copyright Citrix Systems, Inc. All rights reserved.

"""CWC Service Key Signing Algorithm.

Use :py:class:`cwc.auth.signing.CwcSigningAlgorithm` to specify the signing algorithm
for CWC service keys.
"""
from cwc.util import isempty

class CwcSigningAlgorithm:
	"""Defines the signing algorithms supported by CC service keys."""

	# signing algorithms
	RSA_SHA256_V1 = 'CC-RSA-SHA256'
	RSA_SHA256_V2 = 'CC-RSA-SHA256-2'

	# min, max, and supported versions
	MIN_VERSION = RSA_SHA256_V1
	MIN_SUPPORTED_VERSION = RSA_SHA256_V1
	MAX_VERSION = RSA_SHA256_V2

	# version history
	_VERSIONS = [RSA_SHA256_V1, RSA_SHA256_V2]

	@staticmethod
	def isvalid(signing_algorithm):
		"""Checks is a given signing algorithm is valid (known).

		:param signing_algorithm:   the algorithm to check
        :returns:                   a boolean value indicating whether the algorithm is valid
		"""
		return not isempty(signing_algorithm) and signing_algorithm.upper() in CwcSigningAlgorithm._VERSIONS

	@staticmethod
	def issupported(signing_algorithm, min_version = None, max_version = None):
		"""Checks if a given signing algorithm is supported.

		:param signing_algorithm:   the algorithm to check
        :param min_version:         the minimum supported version (optional)
        :param max_version:			the maximum supported version (optional)
        :returns:                   a boolean value indicating whether the algorithm is supported
        """
		if isempty(signing_algorithm):
			return False

		versions = CwcSigningAlgorithm._VERSIONS
		index = versions.index(signing_algorithm.upper())

		if index < 0:
			return false

		min = versions.index(CwcSigningAlgorithm.MIN_SUPPORTED_VERSION) if isempty(min_version) else versions.index(min_version.upper())
		max = versions.index(CwcSigningAlgorithm.MAX_VERSION) if isempty(max_version) else versions.index(max_version.upper())
		return min >= 0 and max >= 0 and min <= index and index <= max

	@staticmethod
	def get_signature_string(*args):
		"""Creates a signature string given a set of parameters.

		:param args:	the list of parameters to sign
        :returns:       the signature string that should be signed
        """
		return '\n'.join(map(str, args))
