#!/usr/bin/python
########################################################################################
#
#  Copyright (c) 2018-2022 Citrix Systems, Inc.
#  All rights reserved.
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions are met:
#      * Redistributions of source code must retain the above copyright
#        notice, this list of conditions and the following disclaimer.
#      * Redistributions in binary form must reproduce the above copyright
#        notice, this list of conditions and the following disclaimer in the
#        documentation and/or other materials provided with the distribution.
#      * Neither the name of the Citrix Systems, Inc. nor the
#        names of its contributors may be used to endorse or promote products
#        derived from this software without specific prior written permission.
#
#  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
#  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
#  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
#  DISCLAIMED. IN NO EVENT SHALL CITRIX SYSTEMS, INC. BE LIABLE FOR ANY
#  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
#  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
#  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
#  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
#  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
#  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
########################################################################################

"""Utilities for provisioning or deprovisioning a NetScaler user account."""

import grp
import os
import pwd
import re
import shutil
import subprocess
import tempfile

from google_compute_engine import constants
from google_compute_engine import file_utils

try:
    from libnitrocli import nitro_cli, nitro_cli_output_parser
except:
    pass


USER_REGEX = re.compile(r'\A[A-Za-z0-9._][A-Za-z0-9._-]*\Z')

class NSVPXAccountsUtils(object):
  """System user account configuration utilities."""

  google_comment = '# Added by Google Agent'

  def __init__(
      self, logger, groups=None, remove=False, groupadd_cmd=None,
      useradd_cmd=None, userdel_cmd=None, usermod_cmd=None):
    """Constructor.

    Args:
      logger: logger object, used to write to SysLog and serial port.
      groups: string, a comma separated list of groups.
      remove: bool, True if deprovisioning a user should be destructive.
      groupadd_cmd: string, command to add a new group.
      useradd_cmd: string, command to create a new user.
      userdel_cmd: string, command to delete a user.
      usermod_cmd: string, command to modify user's groups.
    """
    try:
        self.__nitro = nitro_cli()
        self.__parser = nitro_cli_output_parser()
    except:
        self.__nitro = None
        self.__parser = None

    self.logger = logger
    self.google_users_dir = '/nsconfig/.GCP/'
    self.google_users_file = os.path.join(self.google_users_dir, 'google_users')
    self.google_created_users_file = os.path.join(self.google_users_dir, 'google_created_users')
    self.remove = remove

  def _GetUser(self, username):
    """Retrieve a NetScaler user account.

    Args:
      user: string, the name of the NetScaler user account to retrieve.

    Returns:
      NetScaler user or None if it does not exist.
    """
    result = self.__nitro.get_systemuser(username)
    #self.logger.debug('result of get systemuser: %s', result)
    if self.__parser.success(result):
        #self.logger.info('Found user %s.', username)
        return username
    else:
        self.logger.info('USER NOT Found %s.', username)
        return None

  def _AddUser(self, username):
    """Configure a NetScaler user account. Append agent created
       users in a separate file google_created_users

    Args:
      user: string, the name of the NetScaler user account to create.

    Returns:
      bool, True if user creation succeeded.
    """
    #self.logger.info('Creating a new user account for %s.', username)
    if not os.path.exists(self.google_users_dir):
      os.makedirs(self.google_users_dir)

    result = self.__nitro.add_systemuser(username)
    #self.logger.debug('result of add systemuser: %s', result)
    if self.__parser.success(result):
        f = open (self.google_created_users_file, "a")
        f.write(username + "\n")
        f.close()
        #self.logger.info('Added user %s.', username)
    else:
        self.logger.info('User add failed for  %s.', username)
        return False

    result = self.__nitro.bind_systemuser(username)
    #self.logger.debug('result of bind systemuser: %s', result)
    if self.__parser.success(result):
        #self.logger.info('Superuser policy set for user %s.', username)
        return True
    else:
        self.logger.info('ERROR:superuser policy set failed for  %s.', username)

  def _UpdateAuthorizedKeys(self, username, ssh_keys):
    """Update the authorized keys file for a NetScaler user with a list of SSH keys.

    Args:
      user: string, the name of the NetScaler user account.
      ssh_keys: list, the SSH key strings associated with the user.

    Raises:
      IOError, raised when there is an exception updating a file.
      OSError, raised when setting permissions or writing to a read-only
          file system.
    """
    pw_entry = self._GetUser(username)
    if not pw_entry:
        return

    ssh_dir = "/var/pubkey/%s/.ssh/" % username
    if not os.path.exists(ssh_dir):
      os.makedirs(ssh_dir)

    # Not all sshd's support multiple authorized_keys files so we have to
    # share one with the user. We add each of our entries as follows:
    #  # Added by Google
    #  authorized_key_entry
    authorized_keys_file = os.path.join(ssh_dir, 'authorized_keys')
    #self.logger.info("Created ssh keyfile: %s", authorized_keys_file)

    # Do not write to the authorized keys file if it is a symlink.
    if os.path.islink(ssh_dir) or os.path.islink(authorized_keys_file):
      self.logger.warning(
          'Not updating authorized keys for user %s. File is a symlink.', username)
      return

    # Create entry in the authorized keys file.
    prefix = 'gcp-'
    with tempfile.NamedTemporaryFile(
        mode='w', prefix=prefix, delete=True) as updated_keys:
      updated_keys_file = updated_keys.name
      if os.path.exists(authorized_keys_file):
        lines = open(authorized_keys_file).readlines()
      else:
        lines = []

      google_lines = set()
      for i, line in enumerate(lines):
        if line.startswith(self.google_comment):
          google_lines.update([i, i+1])


      # Write user's authorized key entries.
      for i, line in enumerate(lines):
        if i not in google_lines and line:
          line += '\n' if not line.endswith('\n') else ''
          updated_keys.write(line)

      # Write the Google authorized key entries at the end of the file.
      # Each entry is preceded by '# Added by Google'.
      for ssh_key in ssh_keys:
        ssh_key += '\n' if not ssh_key.endswith('\n') else ''
        self.logger.debug('Adding key %s and google line.', (ssh_key[:50]+"**********"))
        updated_keys.write('%s\n' % self.google_comment)
        updated_keys.write(ssh_key)


      # Write buffered data to the updated keys file without closing it and
      # update the NetScaler's authorized keys file.
      updated_keys.flush()
      shutil.copy(updated_keys_file, authorized_keys_file)

    file_utils.SetPermissions(
        authorized_keys_file, mode=0o600)

  def _RemoveAuthorizedKeys(self, username):
    """Remove a NetScaler user account's authorized keys file to prevent login.

    Args:
      user: string, the NetScaler user account to remove access.
    """
    pw_entry = self._GetUser(username)
    if not pw_entry:
      return

    home_dir = pw_entry.pw_dir
    authorized_keys_file = os.path.join(home_dir, '.ssh', 'authorized_keys')
    if os.path.exists(authorized_keys_file):
      try:
        os.remove(authorized_keys_file)
      except OSError as e:
        message = 'Could not remove authorized keys for user %s. %s.'
        self.logger.warning(message, user, str(e))

  def GetConfiguredUsers(self):
    """Retrieve the list of configured Google user accounts.

    Returns:
      list, the username strings of users congfigured by Google.
    """
    if os.path.exists(self.google_users_file):
      users = open(self.google_users_file).readlines()
    else:
      users = []
    return [user.strip() for user in users]

  def SetConfiguredUsers(self, users):
    """Set the list of configured Google user accounts.

    Args:
      users: list, the username strings of the NetScaler accounts.
    """
    prefix = 'gcp-'
    with tempfile.NamedTemporaryFile(
        mode='w', prefix=prefix, delete=True) as updated_users:
      updated_users_file = updated_users.name
      for user in users:
        updated_users.write(user + '\n')
      updated_users.flush()
      if not os.path.exists(self.google_users_dir):
        os.makedirs(self.google_users_dir)
      shutil.copy(updated_users_file, self.google_users_file)

    file_utils.SetPermissions(self.google_users_file, mode=0o600, uid=0, gid=0)

  def UpdateUser(self, username, ssh_keys):
    """Update a NetScaler user with authorized SSH keys.

    Args:
      user: string, the name of the NetScaler user account.
      ssh_keys: list, the SSH key strings associated with the user.

    Returns:
      bool, True if the user account updated successfully.
    """
    save_config = False
    if not bool(USER_REGEX.match(username)):
      self.logger.warning('Invalid user account name %s.', username)
      return False

    if not self._GetUser(username):
      if not (self._AddUser(username)):
        self.logger.warning('Add user returning early. Keys:')
        for ssh_key in ssh_keys:
            self.logger.warning('%s', (ssh_key[:50]+"**********"))
        return False
      else:
        save_config = True

    try:
      self.logger.warning('****KEYS:')
      for ssh_key in ssh_keys:
          self.logger.warning('%s.', (ssh_key[:50]+"**********"))
      self._UpdateAuthorizedKeys(username, ssh_keys)
    except (IOError, OSError) as e:
      message = 'Could not update the authorized keys file for user %s. %s.'
      self.logger.warning(message, username, str(e))
      return False
    else:
      return True

  def RemoveUser(self, username):
    """Remove a NetScaler user account.

    Args:
      user: string, the NetScaler user account to remove.
    """
    """ Check if user is already present in the file.
        If so, he is an user created by the agent. Remove the user
        and remove his authorized keys file. 
        If not, he is an user created through nscli. Keep it intact.
        In both cases, any Google Added SSH keys should be removed
        Only user added SSH keys should be retained
    """
    remove_user = False
    self.logger.info('Removing Google Agent added SSH keys for %s.', username)

    ssh_dir = "/var/pubkey/%s/.ssh/" % username

    # Not all sshd's support multiple authorized_keys files so we have to
    # share one with the user. We add each of our entries as follows:
    #  # Added by Google
    #  authorized_key_entry
    authorized_keys_file = os.path.join(ssh_dir, 'authorized_keys')

    # Create entry in the authorized keys file.
    prefix = 'gcp-'
    with tempfile.NamedTemporaryFile(
        mode='w', prefix=prefix, delete=True) as updated_keys:
      updated_keys_file = updated_keys.name
      if os.path.exists(authorized_keys_file):
        lines = open(authorized_keys_file).readlines()
      else:
        lines = []

      google_lines = set()
      for i, line in enumerate(lines):
        if line.startswith(self.google_comment):
          google_lines.update([i, i+1])


      # Write user's authorized key entries if it was present
      for i, line in enumerate(lines):
        if i not in google_lines and line:
          line += '\n' if not line.endswith('\n') else ''
          updated_keys.write(line)

      # Write buffered data to the updated keys file without closing it and
      # update the NetScaler's authorized keys file.
      updated_keys.flush()
      shutil.copy(updated_keys_file, authorized_keys_file)

    file_utils.SetPermissions(
        authorized_keys_file, mode=0o600)

    if not os.path.exists(self.google_created_users_file):
        self.logger.info('%s file is missing. No users added by Google Agent so far',self.google_created_users_file)
        return
    else:
        f = open(self.google_created_users_file,"r+")
        d = f.readlines()
        f.seek(0)
        for i in d:
            if i.rstrip() != username:
                f.write(i) #Skip the entry if it is present
            else:
                self.logger.info('User added by agent. We need to remove.')
                remove_user = True
        f.truncate()
        f.close()

    if remove_user:
        if self.remove:
            result = self.__nitro.del_systemuser(username)
            #self.logger.debug('result of set systemuser: %s', result)
            if self.__parser.success(result):
                #self.logger.info('Removed user account %s !!', username)
                pass
            else:
                self.logger.info('Could not remove user %s. %s.', username)
        self._RemoveAuthorizedKeys(username)
    else:
        self.logger.info('Entry not found. User not created by Google Agent. %s.', username)

  def _SaveNSConfig(self):
    """Save ns.conf after any operation.

    Args:
     None.
    """
    result = self.__nitro.save_nsconfig()
    if self.__parser.success(result):
        #self.logger.info('ns.conf saved successfully!!')
        return True
    else:
        self.logger.info('ERROR: ns.conf could not saved')
        return False

