# FILE:         $Id: //depot/main/TOT/usr.src/netscaler/safenet/install_client.sh#3 $
# LAST CHECKIN: $Author: hongch $
#               $DateTime: 2019/01/10 15:46:37 $

#!/usr/bin/bash

# Copyright 2016-2019 Citrix Systems, Inc. All rights reserved.
# This software and documentation contain valuable trade
# secrets and proprietary property belonging to Citrix Systems, Inc.
# None of this software and documentation may be copied,
# duplicated or disclosed without the express
# written permission of Citrix Systems, Inc.

set -e
# Setting the current directory to be /var/safenet
cd /var/safenet

###constants
SAFENET_VER="600"
SAFENET_TAR="safenet_dirs.tar"
VER_FILE="safenet/version"
SAFENET_CURR_VER=""
OLD="0"
BACKUP_CREATED="0"
SAFENET_DIR="safenet"
CURR_SAFENET_DIR="safenet/"
SAFENET_OLD="safenet_old_"
CONF_FILE="/etc/Chrystoki.conf"
BACKUP_VER_FILE="/version"
ORG_SAFENET_FILE="SAClient_"
LIB_CRYPTO="/usr/lib/libcrypto2_64.so"
LOGFILE="installation.log"
#### end of constants

## functions
logger()
{
	echo "$@"
	LOGTIME=$(date +"%F %T")
	echo "${LOGTIME} : $@" >>${LOGFILE}
}
usage()
{
	echo "usage: $0 [[-v 600 | -v 622 | -v 722 | -v 1030 ] [ -o]] | [ -h ]"
	echo "SafeNet client software versions 6.0.0, 6.2.2, 7.2.2 and 10.3.0 are supported."
	echo "e.g $0 -v 600    -- will load fresh copy of SafeNet client software version 6.0.0"
	echo "e.g $0 -v 622 -o -- will load the old copy of SafeNet Client Software verison 6.2.2"
}

backup()
{
	# Save the chrystoki.conf file with chrystoki_version.conf"
	# Log the safenet version and time
	# Unlink the symlink for libcrypto2_64.so
	# Move the old backup
	# Create the new backup
	# Remove the safenet directory
	# Remove the old backup
	# Load the old/New Client libraries.

	# Creating tar file name
	SAFENET_BACKUP="${SAFENET_OLD}${SAFENET_CURR_VER}.tgz"
	SAFENET_BACKUP_CP="${SAFENET_BACKUP}.copy"
	CONF_FILE_BACKUP="${SAFENET_DIR}/Chrystoki_${SAFENET_CURR_VER}.conf"

	# Save the chrystoki.conf file with chrystoki_version.conf"
	if [ -e "${CONF_FILE}" ]; then
		if ! cp "${CONF_FILE}" "${CONF_FILE_BACKUP}" ; then
			logger "ERROR: Failed to copy the Safenet config file."
		fi
	fi

	# Log the safenet version and time
	echo "${SAFENET_VER}" > "${SAFENET_DIR}${BACKUP_VER_FILE}"
	RIGHT_NOW=$(date +"%x %r %Z")
	echo "Time of Backup is ${RIGHT_NOW}" >> "${SAFENET_DIR}${BACKUP_VER_FILE}"

	#Unlink the symlink for libcrypto2_64.gz
	if [ -e "${LIB_CRYPTO}" ]; then
		if  ! rm -rf "${LIB_CRYPTO}" ; then
			logger "ERROR: Cannot find ${LIB_CRYPTO} file."
			exit 1
		else
			logger "Removed the symlink to ${LIB_CRYPTO} file."
		fi
	fi

	#tar the backup directory.
	#check if the tar file is present, then move old the tar file for same version
	if [ -f "${SAFENET_BACKUP}" ]; then
		if ! mv "${SAFENET_BACKUP}" "${SAFENET_BACKUP_CP}" ; then
			logger "ERROR: Failed to copy the backup file ${SAFENET_BACKUP} to ${SAFENET_BACKUP_CP}"
			exit 1
		fi
		logger "Saved the backup file ${SAFENET_BACKUP} as ${SAFENET_BACKUP_CP}"
		BACKUP_CREATED="1"
	fi

	#create the new tar file.
	if  tar czf "${SAFENET_BACKUP}" "${CURR_SAFENET_DIR}" ; then
		logger "Created a ${SAFENET_BACKUP} file."
		if [ -d "${SAFENET_DIR}" ]; then
			if ! rm -rf "${CURR_SAFENET_DIR}" ; then
				logger "ERROR: Failed to delete the safenet directory."
				exit 1
			else
				logger "Removed Safenet directory."
			fi
		fi
	else
		logger "ERROR: Failed to create a tar file."
		exit 1
	fi

}

create_new()
{
	LOAD_SAFENET="${ORG_SAFENET_FILE}${SAFENET_VER}.tgz"
	if [ -f "${SAFENET_TAR}" ]; then
		if tar -xpf "${SAFENET_TAR}" -C /var "safenet/${LOAD_SAFENET}"; then
			logger "Extracted ${LOAD_SAFENET} file from ${SAFENET_TAR}."
		else
			logger "ERROR: Failed to extract ${LOAD_SAFENET} file from ${SAFENET_TAR}."
		fi
	fi

	# untar the given version of tar file.
	if [ -f "${LOAD_SAFENET}" ]; then
		logger "Starting to extract ${LOAD_SAFENET} file."
		#untar the file
		if ! tar -xzf "${LOAD_SAFENET}" ; then
			logger "ERROR: Failed to extract ${LOAD_SAFENET} file."
			exit 1
		fi
		logger "Extracted ${LOAD_SAFENET} file."
	else
		logger "ERROR: Couldn't find ${LOAD_SAFENET} file."
		exit 1
	fi

	if [ -f "${LOAD_SAFENET}" ]; then
		if rm "${LOAD_SAFENET}" ; then
			logger "Removing ${LOAD_SAFENET} file."
		fi
	fi
}

load_old()
{
	LOAD_OLD_SAFENET="${SAFENET_OLD}${SAFENET_VER}.tgz"
	if [ -f "${LOAD_OLD_SAFENET}" ]; then
		if [ "${SAFENET_VER}" == "${SAFENET_CURR_VER}" ] && [ "${BACKUP_CREATED}" == "1" ]; then
			LOAD_OLD_SAFENET="${SAFENET_OLD}${SAFENET_VER}.tgz.copy"
		fi
		#untar the old file
		if ! tar xzf "${LOAD_OLD_SAFENET}" ; then
			logger "ERROR: Failed to extract ${LOAD_OLD_SAFENET} file."
			exit 1
		else
			logger "Extracted ${LOAD_OLD_SAFENET} file"
		fi
	else
		logger "ERROR: Couldn't find ${LOAD_OLD_SAFENET} file, reverting to running ver."
		SAFENET_BACKUP="${SAFENET_OLD}${SAFENET_CURR_VER}.tgz"
		if [ -f "${SAFENET_BACKUP}" ]; then
			if ! tar -xzf "${SAFENET_BACKUP}" ; then
				logger "ERROR: Failed to revert to old config."
				exit 1;
			fi
		fi
		exit 1
	fi
}

### Main

## read command line arguments

# use -gt 1 to consume two arguments per pass in the loop
# (e.g. each argument has a corresponding value to go with it.)
while [ $# -gt 0 ]
do
	key="$1"
	case $key in
		-h|-\?|--help)					usage
								exit 1
								;;
		-v|--version)				 	if [ -n "$2" ]; then
								   SAFENET_VER="$2"
								   shift
								fi
								;;
		-o|--old)					OLD="1"
								;;
		*)						logger "Unkown param \"$key\""
								usage
								exit 1
								;;
	esac
	shift
done

#get the current running version of safenet
if [ ! -f "${VER_FILE}" ]; then
	SAFENET_CURR_VER=${SAFENET_VER}
else
	SAFENET_CURR_VER=$(head -n 1 "${VER_FILE}")
fi

logger "*****************************************************************************"
logger "Current Version: ${SAFENET_CURR_VER}"
logger "Installing Version: ${SAFENET_VER}"

#Check the version provided on CLI arguments
if [ "${SAFENET_VER}" != "600" ] && [ "${SAFENET_VER}" != "622" ] && [ "${SAFENET_VER}" != "722" ] && [ "${SAFENET_VER}" != "1030" ] && [ "${SAFENET_VER}" != "1080_linux"]; then
	logger "ERROR: Please provide -v 600, -v 622, -v 722 or -v 1030 or -v 1080_linux"
	exit 1
fi

if [ -d "${SAFENET_DIR}" ]; then
	backup
fi

if [ "${OLD}" == "0" ]; then
	create_new
else
	load_old
fi

#remove copy of old tar file
SAFENET_BACKUP_CP="${SAFENET_OLD}${SAFENET_CURR_VER}.tgz.copy"
if [ -f "${SAFENET_BACKUP_CP}" ]; then
	if ! rm -rf "${SAFENET_BACKUP_CP}" ; then
		logger "ERROR: Failed to remove the backup copy file ${SAFENET_BACKUP_CP}"
		exit 1
	fi
	logger "Removed a back-up copy file ${SAFENET_BACKUP_CP}."
fi

logger "*****************************************************************************"
logger "Now follow the configuration steps document available online on Citrix edocs."
logger "*****************************************************************************"
