#!/usr/bin/env python
"""
Copyright 2000-2025 Citrix Systems, Inc. All rights reserved.
This software and documentation contain valuable trade secrets
and proprietary property belonging to Citrix Systems, Inc.
None of this software and documentation may be copied,
duplicated or disclosed without the express written permission
of Citrix Systems, Inc.
"""


import sys
import os
import json
import inspect
import signal
from collections import OrderedDict

import rainman_core.common.constants as CONST
from rainman_core.common.logger import RainLogger
from rainman_core.common.exception import *
from rainman_core.common.ns_errors import *
from rainman_core.common.rain_protocol_mapping import *
from rainman_core.common import rain
from rainman_core.common.rain_globals import init_rain_globals
from rainman_core.common.ipc_lock import get_conf_lock

rlog = RainLogger(CONST.CONFIG_LOG_FILE_NAME, CONST.DEFAULT_LOG_LEVEL)
log = rlog.logger
config, local, cloud = init_rain_globals()
MAX_RES_LIST_LEN = 4095
MAX_RES_COUNT = 1024 # max number of cp/asg/cvip to display

def add_alarm(name, alarmtype, metric, threshold, comparison, period, action):
    alarm = rain.alarm()
    try:
        alarm = alarm.get(config, name)
    except config_not_present:
        alarm.name = name
        alarm.alarmtype = alarmtype
        alarm.metric = metric
        alarm.threshold = threshold
        alarm.comparison = comparison
        alarm.period = period
        alarm.action = action
        alarm.add(config, alarm)
        log.info(f"alarm {name!r} added")
    else:
        log.warning(f"alarm {name!r} already exists")
        print(json.dumps({'response': 'done', 'errno': 0, 'errmsg': ""}))

    print(json.dumps({'response': 'done', 'errno': 0, 'errmsg': ""}))


def remove_alarm(name):
    alarm = rain.alarm()
    log.info("Removing alarm %s" % (name))

    try:
        alarm = alarm.get(config, name)
    except config_not_present:
        log.info("Alarm %s does not exist" % (name))
    else:
        alarm.remove(config, alarm)

    print(json.dumps({'response': 'done', 'errno': 0, 'errmsg': ""}))


def add_lb(name, protocol, ip, port):
    lb = rain.loadbalancer()
    log.info("Adding load balancer %s" % (name))

    try:
        lb = lb.get(config, name)
    except config_not_present:
        lb.name = name
        lb.lbtype = "vanilla"
        lb.ip = ip
        lb.port = port
        lb.protocol = protocol
        lb.add(config, lb)
    else:
        log.warning("Load balancer %s already exists" % (lb.name))

    try:
        local.add_lb(lb)
    except config_failed:
        log.critical("Failed to add load balancer %s, aborting." % (name))
        lb.remove(config, lb)

    return lb


def add_azuretag(tag_name, tag_value):
    azuretag = rain.azuretag()
    azuretag.set(tag_name, tag_value)
    azuretagname = azuretag.name
    add_aztag = False

    log.info("Adding azuretag %s" % (azuretagname))

    try:
        azuretag = azuretag.get(config, azuretag)
    except config_not_present:
        azuretag.set(tag_name, tag_value)
        add_aztag = True

    if add_aztag == True:
        azuretag.add(config, azuretag)
    return azuretag


def add_service(group_name, service_name, port, protocol, cloud_profile_type, azure_tag=None, enabled=True):
    log.info("Adding service %s" % (service_name))
    service = rain.service()

    try:
        service = service.get(config, service_name)
    except config_not_present:
        service.name = service_name
        service.group_name = group_name
        service.port = port
        service.protocol = protocol
        service.cloud_profile_type = cloud_profile_type
        if azure_tag != None:
            service.azuretag = azure_tag.name
        service.enabled = enabled
        service.add(config, service)
        log.info("service added.")
    else:
        log.warning("service %s already exists." % service_name)

    return service


def add_notification(group):
    try:
        if cloud.check_event_queue():
            queue_conf = create_event_queue()
            if queue_conf is not None:
                cloud.configure_events_for_group(queue_conf, group)
    except Exception:
        rlog.error_trace(f"Exception seen while configuring notification for autoscale group {group.name}")


def add_group(grp_name, service_name, protocol, port, drain, drain_time, group_type, cloud_profile_type, azure_tag=None, poll_period=None, zone=None, rg=None):
    log.info("Adding group %s" % (grp_name))
    group = rain.group()
    try:
        group = group.get(config, grp_name)
    except config_not_present:
        group.name = grp_name
        group.zone = zone
        group.rg = rg
        group.protocol = protocol  # keep for backward compatibility
        group.port = port
        group.drain = drain
        group.drain_time = drain_time
        group.cloud_profile_type = cloud_profile_type
        if azure_tag != None:
            group.azuretag = azure_tag.name
        if poll_period != None:
            group.poll_period = poll_period
        group.add(config, group)
        add_notification(group)
    else:
        log.warning("Group %s already exists, updating group specific params." %
                    (group.name))
        #group.protocol = protocol
        #group.port = port
        group.drain = drain
        group.drain_time = drain_time
        #group.cloud_profile_type = cloud_profile_type
        # if azure_tag != None:
        #    group.azuretag = azure_tag.name
        if poll_period != None:
            group.poll_period = poll_period

    service = add_service(grp_name, service_name, port,
                          protocol, cloud_profile_type, azure_tag)
    if group_type != 'SERVICEGROUP':
        try:
            local.add_group(service)
        except config_failed:
            # Roll rainman config back if local config fails
            service.remove(config, service)
            group.remove(config, group)
            raise config_failed("failed to add servicegroup %s" % service_name)

    return (group, service)


def add_binding(profile_name, group, service, lb, cloud_profile_type, alarms=None):

    binding = rain.group_lb_alarm_binding()

    try:
        binding = binding.get(config, profile_name)
    except config_not_present:
        # No bindings exist at all for this lb
        binding.name = profile_name
        binding.lb_name = lb.name
        binding.group = group.name
        binding.service = service.name
        binding.type = cloud_profile_type
        binding.alarms = alarms
        binding.add(config, binding)
        log.info("Create new binding, profile_name=%s, %s to %s with %s" %
                 (profile_name, lb.name, group.name, str(binding.alarms)))
    else:
        log.warning("Local binding %s: %s to %s already exists" %
                    (binding.name, binding.lb_name, binding.group))

    # 5. Finally create the local binding, ex. NS
    try:
        local.add_group_to_lb(service, lb)
    except config_failed as e:
        # The binding failed on local, delete rainman config entry too.
        binding.remove(config, binding)
        # But throw an exception too because this shouldnt happen
        raise

    return binding


def _resolve_g_sg_names(expected_grpname, group_port):
    result = None
    err_msg = None
    if not expected_grpname:
        return (None, None, None)
    group_name, service_name, err_msg = rain.group.get_group_service_names(
        config, expected_grpname, group_port)
    if not group_name or not service_name:
        errno = NS_BASE_ERRORS['NSERR_INVAL']['errno']
        desc = NS_BASE_ERRORS['NSERR_INVAL']['errno']
        if err_msg:
            desc = f"{desc}: {err_msg}"
        result = {'response': 'failed', 'errno': errno, 'errmsg': desc}
    return (result, group_name, service_name)


def _resolve_g_name(expected_grpname):
    result = None
    err_msg = None
    if not expected_grpname:
        return (None, None, None)
    group_name, service_name, err_msg = rain.group.get_group_service_names(
        config, expected_grpname)
    if not group_name:
        errno = NS_BASE_ERRORS['NSERR_INVAL']['errno']
        desc = NS_BASE_ERRORS['NSERR_INVAL']['errno']
        if err_msg:
            desc = f"{desc}: {err_msg}"
        result = {'response': 'failed', 'errno': errno, 'errmsg': desc}
    return (result, group_name, service_name)


def add_cloudprofile_by_tag(profile_name, lb_name, expected_grpname, lb_protocol, lb_ip, lb_port, group_protocol, group_port, tag_name, tag_value, period):
    try:
        res, group_name, service_name = _resolve_g_sg_names(
            expected_grpname, group_port)
        if res:
            print(json.dumps(res))
            return

        result = check_cp_not_present(profile_name, CONST.AZURETAGS, lb_name, lb_protocol, lb_port,
                                      lb_ip, group_name, service_name, group_protocol, group_port, tag_name, tag_value)

        if result:
            print(json.dumps(result))
            return

        if cloud.get_cloud_platform() not in ('AZURE', 'AZURESTACK'):
            errno = NS_BASE_ERRORS['NSERR_NO_PERM_ON_PLATFORM']['errno']
            desc = NS_BASE_ERRORS['NSERR_NO_PERM_ON_PLATFORM']['desc']
            print(json.dumps({'response': 'failed',
                              'errno': errno, 'errmsg': desc}))
            return

        lb = None
        group = None
        service = None

        azuretag = add_azuretag(tag_name, tag_value)
        lb = add_lb(lb_name, lb_protocol, lb_ip, lb_port)
        group, service = add_group(group_name, service_name, group_protocol, group_port,
                                   False, 60, 'CLOUDPROFILE', CONST.AZURETAGS, azuretag, period)
        add_binding(profile_name, group, service, lb, CONST.AZURETAGS)

    except config_failed as e:
        log.warn("cloud profile creation using azure tags failed, rolling back")
        try:
            binding = rain.group_lb_alarm_binding().get(config, profile_name)
            _remove_binding(binding)
        except:
            pass
        _remove_lb(lb_name)
        _remove_service(service_name)
        _remove_group(group_name)

        if azuretag:
            try:
                azuretag.remove(config, azuretag)
            except:
                pass

        errno = str(e).split()[0].rstrip(',')
        errmsg = ' ' .join(str(e).split()[1:])
        print(json.dumps({'response': 'failed',
                          'errno': errno, 'errmsg': errmsg}))
        return

    print(json.dumps({'response': 'success', 'errno': 0,
                      'errmsg': "Cloudprofile by tag added."}))


def add_cloudprofile(profile_name, lb_name, expected_grpname, lb_protocol, lb_ip, lb_port, group_protocol, group_port, drain, drain_time, alarm_names=None):
    try:
        cloud.check_privileges("appautoscale")
    except:
        rlog.error_trace("Exception seen while cloud.check_privileges")
        pass
    try:
        res, group_name, service_name = _resolve_g_sg_names(
            expected_grpname, group_port)
        if res:
            print(json.dumps(res))
            return

        result = check_cp_not_present(profile_name, CONST.AUTOSCALE, lb_name, lb_protocol,
                                      lb_port, lb_ip, group_name, service_name, group_protocol, group_port)
        if result:
            print(json.dumps(result))
            return

        result, zone, rg = check_cloud_asg_exist(group_name)
        if result:
            print(json.dumps(result))
            return

        log.debug("add cloud profile: pre-checks done.")
        lb = None
        group = None
        service = None

        # 1. Add lb to rainman
        lb = add_lb(lb_name, lb_protocol, lb_ip, lb_port)

        # For AZURE, implicit 60 sec is set for graceful deletion
        if cloud.get_cloud_platform() in ('AZURE', 'AZURESTACK'):
            drain_time = 60

        # 2. Check and add groups to rainman
        group, service = add_group(group_name, service_name, group_protocol, group_port,
                                   drain, drain_time, 'CLOUDPROFILE', CONST.AUTOSCALE, zone=zone, rg=rg)
        add_binding(profile_name, group, service, lb, CONST.AUTOSCALE, alarm_names)
        print(json.dumps({'response': 'success',
                          'errno': 0, 'errmsg': "Cloudprofile added."}))

        if cloud.get_cloud_platform() == 'GCP':
            try:
                signal_rain_scale(signal.SIGUSR2)
            except:
                rlog.error_trace("failed to signal rain scale")
        log.info("add cloudprofile completed")
        return
    except config_failed as e:
        log.warn("cloud profile creation failed, rolling back")
        errno = str(e).split()[0].rstrip(',')
        errmsg = ' ' .join(str(e).split()[1:])
        print(json.dumps({'response': 'failed',
                          'errno': errno, 'errmsg': errmsg}))
    except:
        rlog.error_trace("exception seen while cloud profile creation")
        errno = NS_BASE_ERRORS['NSERR_IO']['errno']
        desc = NS_BASE_ERRORS['NSERR_IO']['desc']
        errmsg = "add cloudprofile failed with exception."
        result = {'response': desc, 'errno': errno, 'errmsg': errmsg}
        print(json.dumps(result))

    try:
        binding = rain.group_lb_alarm_binding().get(config, profile_name)
        _remove_binding(binding)
    except:
        pass
    _remove_lb(lb_name)
    _remove_service(service_name)
    _remove_group(group_name)
    log.info("add cloudprofile roll back complete.")


def check_sg_not_configured(service_name):
    result = None
    errmsg = None

    try:
        sg = rain.service().get(config, service_name)
        errmsg = f"servicegroup {service_name} exist"
    except config_not_present:
        pass

    if errmsg:
        errno = NS_BASE_ERRORS['NSERR_EXIST']['errno']
        desc = NS_BASE_ERRORS['NSERR_EXIST']['desc']
        result = {'response': desc, 'errno': errno, 'errmsg': errmsg}
    return result


def check_g_sg_configured(group_name=None, service_name=None, port=None, protocol=None):
    result = None
    errmsg = None

    if group_name:
        try:
            group = rain.group().get(config, group_name)
        except config_not_present:
            errmsg = f"group {group_name} doesnt exist"

    if service_name:
        try:
            sg = rain.service().get(config, service_name)
            if port and port != sg.port:
                errmsg = f"servicegroup {service_name!r} already exist with port {sg.port!r}"
            elif protocol and protocol != sg.protocol:
                errmsg = f"servicegroup {service_name!r} already exist with protocol{sg.protocol!r}"
        except config_not_present:
            errmsg = f"servicegroup {service_name} doesnt exist"

    if errmsg:
        errno = NS_BASE_ERRORS['NSERR_SVC_GRP']['errno']
        desc = NS_BASE_ERRORS['NSERR_SVC_GRP']['desc']
        result = {'response': desc, 'errno': errno, 'errmsg': errmsg}
    return result


def check_cp_not_present(profile_name, profile_type, lb_name, lb_protocol, lb_port, lb_ip, group_name, service_name, group_protocol, group_port, tag_name=None, tag_value=None):
    result = None
    check_list = ['profile', 'lb', 'service', 'azuretag', 'ip']
    errmsg = None

    for component in check_list:
        try:
            if component == 'profile':
                binding = rain.group_lb_alarm_binding().get(config, profile_name)
                errmsg = f'Cloud profile: {profile_name!r} alreay exists'
                break
            if component == 'lb':
                lb = rain.loadbalancer().get(config, lb_name)
                errmsg = f'Lb vserver: {lb_name!r} alreay exists'
                break
            if component == 'service':
                sg = rain.service().get(config, service_name)
                if sg and group_port != sg.port:
                    errmsg = f"servicegroup {sg.name!r} with different port {sg.port!r} already exists"
                elif group_protocol != sg.protocol:
                    errmsg = f"servicegroup {sg.name!r} with different protocol {sg.protocol!r} already exists"
                elif profile_type != sg.cloud_profile_type:
                    errmsg = f"servicegroup {sg.name!r} with different cloud profile type {sg.cloud_profile_type.name!r} already exists"
                elif not sg.enabled:
                    errmsg = f"servicegroup exists with autoscale for same port and protocol, cloud profile operation for servicegroup is restricted"
                else:
                    log.warn(
                        f'servicegroup {service_name} exists, not blocking cloudprofile check')
                    continue
                break
            if component == 'azuretag':
                if (profile_type == CONST.AZURETAGS):
                    azuretag = rain.azuretag().get(config, tag_name+tag_value)
                    errmsg = 'Azure tag name ' + tag_name + ' with value ' + tag_value + ' exists'
                    break
            if component == 'ip':
                lbs = rain.loadbalancer().get(config)
                match_lb = [lb for lb in lbs if lb_port ==
                            lb.port and lb_ip == lb.ip]
                if match_lb:
                    errmsg = f"The combination of protocol {lb_protocol!r} and port {lb_port!r} and ipaddr {lb_ip!r} already exists"
                    break
        except config_not_present:
            pass

    if errmsg:
        errno = NS_BASE_ERRORS['NSERR_EXIST']['errno']
        desc = NS_BASE_ERRORS['NSERR_EXIST']['desc']
        result = {'response': desc, 'errno': errno, 'errmsg': errmsg}

    return result


def check_cloud_asg_exist(group_name):
    result = None
    groups = []
    group_found = False
    zone = None
    rg = None

    try:
        groups = cloud.get_group_info(group_name)
    except (aws_authetication_failure, azure_authentication_failure) as err:
        rlog.error_trace()
        errno = NS_BASE_ERRORS['NSERR_CLOUD_CRED']['errno']
        desc = NS_BASE_ERRORS['NSERR_CLOUD_CRED']['desc']
        result = {'response': desc, 'errno': errno, 'errmsg': f"{err}"}
    except config_failed as e:
        errno = NS_BASE_ERRORS['NSERR_MISSING_POLICY']['errno']
        desc = NS_BASE_ERRORS['NSERR_MISSING_POLICY']['desc']
        errmsg = f'Cannot find scale down(aka scale out) policies in: {group_name!r} in cloud'
        result = {'response': desc, 'errno': errno, 'errmsg': errmsg}
    except:
        rlog.error_trace()
        errno = NS_BASE_ERRORS['NSERR_SVC_GRP']['errno']
        desc = NS_BASE_ERRORS['NSERR_SVC_GRP']['desc']
        errmsg = f'Exception seen while looking for autoscaling group/setting: {group_name!r} in cloud'
        result = {'response': desc, 'errno': errno, 'errmsg': errmsg}
    else:
        try:
            for group in groups:
                if group.name == group_name:
                    group_found = True
                    if cloud.get_cloud_platform() == 'GCP':
                        zone = group.zone
                    if cloud.get_cloud_platform() in ('AZURE', 'AZURESTACK'):
                        rg = group.locations

            if group_found == False:
                errno = NS_BASE_ERRORS['NSERR_SVC_GRP']['errno']
                desc = NS_BASE_ERRORS['NSERR_SVC_GRP']['desc']
                errmsg = f'Cannot find autoscaling group/setting: {group_name!r} in cloud'
                result = {'response': desc, 'errno': errno, 'errmsg': errmsg}
        except:
            rlog.error_trace()
            if not errmsg:
                errmsg = "cloudprofile check operation failed with exception."
                errno = NS_BASE_ERRORS['NSERR_EXIST']['errno']
                desc = NS_BASE_ERRORS['NSERR_EXIST']['desc']
                result = {'response': desc, 'errno': errno, 'errmsg': errmsg}

    return result, zone, rg


def get_group_count_in_group_lb_alarm_binding(group_name):
    bindings = []
    group_cnt = 0

    try:
        bindings = rain.group_lb_alarm_binding().get(config)
    except config_not_present:
        pass
    else:
        for binding in bindings:
            if binding.group == group_name:
                group_cnt += 1

    return group_cnt


def _remove_binding(bindingObj):
    try:
        lb = rain.loadbalancer().get(config, bindingObj.lb_name)
        service = rain.service().get(config, bindingObj.service)
        local.remove_group_from_lb(service, lb)
    except:
        rlog.error_trace(f"failed to remove group {bindingObj.service!r} from lb {bindingObj.lb_name}")
    bindingObj.remove(config, bindingObj)


def _remove_lb(lb_name):
    try:
        lb = rain.loadbalancer().get(config, lb_name)
    except config_not_present as e:
        log.warn(f"lb {lb_name} is missing in rainman config.")
        return
    if lb:
        try:
            local.remove_lb(lb)
        except config_failed as err:
            log.warn("failed to remove load balancer %s : %s" %
                     (lb.name, str(err)))
        lb.remove(config, lb)


def _remove_service(service_name):
    if not service_name:
        log.warn("asked to remove servicegroup without providing name.")
        return
    try:
        service = rain.service().get(config, service_name)
    except config_not_present as e:
        log.warn(f"servicegroup {service_name} is missing in rainman config.")
        return
    if service:
        # block servicegroup deletion if servicegroup is part of present bindings/CP
        try:
            bindings = rain.group_lb_alarm_binding().get(config)
            for binding in bindings:
                if binding.service == service_name:
                    log.warn(f"servicegroup {service_name} is part of a cloud profile. skip deletion")
                    return
        except config_not_present:
            pass
    
        log.info(f"removing service: {service.name!r}")
        service.remove(config, service)
        try:
            server_list = local.get_servers_in_group(service)
            for server in server_list:
                try:
                    local.remove_server_from_group(server, service)
                except:
                    rlog.error_trace("Exception seen while cleaning servicegroup")
            local.remove_group(service)
        except config_failed as err:
            log.warn(f"failed to clean service {service.name!r} : {err!r}")


def _remove_group(group_name, tag=False):
    try:
        group = rain.group().get(config, group_name)
    except config_not_present as e:
        log.warn(f"group {group_name} is missing in rainman config.")
        return
    if group:
        try:
            group_cnt = get_group_count_in_group_lb_alarm_binding(group.name)
            active_services = group.list_all_services_in_group(config)
            if group_cnt or active_services:
                log.info(
                    f"autoscale {group.name} is in {group_cnt} cloud profiles and have {len(active_services)} servicegroups linked, remove restricted.")
            else:
                if tag:
                    azuretag = rain.azuretag().get(config, group.azuretag)
                    if azuretag:
                        azuretag.remove(config, azuretag)
                else:
                    try:
                        cloud.cleanup_on_group_remove(group)
                    except:
                        rlog.error_trace("exception seen while cloud cleanup")
                    try:
                        if cloud.check_event_queue():
                            queue_conf = get_event_queue()
                            if queue_conf is not None:
                                cloud.remove_notification_config_from_group(
                                    queue_conf, group.name)
                            group.remove(config, group)
                            delete_event_queue()
                            log.info(
                                f"autoscale {group.name} and notification config removed")
                            return
                    except:
                        log.warn(
                            "exception seen while notification config cleanup")
                        rlog.error_trace()
                group.remove(config, group)
                log.info(f"autoscale {group.name} removed")
        except:
            rlog.error_trace(f"Failed to remove group {group.name!r}")


def remove_cloudprofile(profile_name):
    binding = None
    lb = None
    group = None
    service = None
    tag = False

    try:
        binding = rain.group_lb_alarm_binding().get(config, profile_name)
        if not binding:
            raise config_not_present(
                f"cloud profile {profile_name!r} is not present")
    except config_not_present as e:
        log.warn(f"cloud profile {profile_name!r} not present.")
        errno = NS_BASE_ERRORS['NSERR_NOENT']['errno']
        desc = NS_BASE_ERRORS['NSERR_NOENT']['desc']
        print(json.dumps(
            {'errno': errno, 'errmsg': str(e), 'response': desc}))
        return

    if int(binding.type) == CONST.AZURETAGS:
        tag = True
        if cloud.get_cloud_platform() not in ('AZURE', 'AZURESTACK'):
            errno = NS_BASE_ERRORS['NSERR_NO_PERM_ON_PLATFORM']['errno']
            desc = NS_BASE_ERRORS['NSERR_NO_PERM_ON_PLATFORM']['desc']
            print(json.dumps({'response': 'failed',
                              'errno': errno, 'errmsg': desc}))
            return
        log.info(f"Deleting cloudprofile {profile_name!r} by azure tag")
    else:
        log.info(f"Deleting cloudprofile {profile_name!r}")

    lb_name = binding.lb_name
    service_name = binding.service
    group_name = binding.group

    try:
        _remove_binding(binding)
        _remove_lb(lb_name)
        _remove_service(service_name)
        _remove_group(group_name, tag)

    except Exception as e:
        print(json.dumps({'response': 'failed', 'errno': str(e).split()[
              0].rstrip(','), 'errmsg': " ".join(str(e).split()[1:])}))
        return

    if tag:
        signal_rain_tags(signal.SIGUSR2)
        print(json.dumps({'response': 'success', 'errno': 0,
                          'errmsg': "Cloudprofile by tag removed."}))
    else:
        print(json.dumps({'response': 'success', 'errno': 0,
                          'errmsg': "Cloudprofile removed."}))


def get_cloudprofile(profile_name=None):
    count = 0
    lb = rain.loadbalancer()
    group = rain.group()
    service = rain.service()
    bindings = []

    try:
        if profile_name == None:
            bindings = rain.group_lb_alarm_binding().get(config)
        else:
            binding = rain.group_lb_alarm_binding().get(config, profile_name)
            bindings.append(binding)
    except config_not_present as e:
        pass

    class group_display_obj(object):

        def __init__(self, profile_name=None, lb=None, group=None, service=None, azuretag=None):
            self.set(profile_name, lb, group, service, azuretag)

        def set(self, profile_name=None, lb=None, group=None, service=None, azuretag=None):
            self.profileName = profile_name if profile_name else None

            if not lb:
                self.vserverName = None
                self.vserverName = None
                self.vserverProtocol = None
                self.vserverPort = None
            else:
                self.vserverName = lb.name
                self.vserverProtocol = int(list(RAIN_PROTOCOLS.keys())[
                                           list(RAIN_PROTOCOLS.values()).index(lb.protocol)])
                self.vserverPort = lb.port
                self.cloudip = lb.ip

            if not group:
                self.drain = None
                self.delay = None
            else:
                self.drain = group.drain
                self.delay = group.drain_time
                self.azurePollPeriod = group.poll_period

            if not service:
                self.asgName = None
                self.serviceProtocol = None
                self.servicePort = None
            else:
                self.asgName = service.name
                self.serviceProtocol = int(list(RAIN_PROTOCOLS.keys())[list(
                    RAIN_PROTOCOLS.values()).index(service.protocol)])
                self.servicePort = service.port
                self.profileType = service.cloud_profile_type

            if not azuretag:
                self.azureTagName = None
                self.azureTagValue = None
            else:
                self.azureTagName = azuretag.tag_name
                self.azureTagValue = azuretag.tag_value

        def toOrderedDict(self):
            cp_ordered = OrderedDict()
            for enum in range(len(CONST.PE_ORDER_GET_CP)):
                if enum not in CONST.PE_ORDER_GET_CP:
                    raise rainman_exception(
                        "index missing from dict keys, please check in get_cloudprofile.")
                cp_ordered[CONST.PE_ORDER_GET_CP[enum]] = self.__dict__[
                    CONST.PE_ORDER_GET_CP[enum]]
            return cp_ordered

    errno = '0'
    errmsg = 'None'
    for binding in bindings:
        try:
            lb = rain.loadbalancer().get(config, binding.lb_name)
            group = rain.group().get(config, binding.group)
            service = rain.service().get(config, binding.service)
            if (group.azuretag is not None):
                azuretag = rain.azuretag().get(config, group.azuretag)
            else:
                azuretag = None
            group_dis = group_display_obj(
                binding.name, lb, group, service, azuretag)
            str_res = json.dumps({'errno': 0, 'errmsg': "None", "response": [group_dis.toOrderedDict()]})
            str_res = str_res.replace("\n", " ")
            if len(str_res) >= MAX_RES_LIST_LEN:
                log.warn(f"Skipping cloudprofile {binding.name!r} from display as output is too long.")
                continue
            print(str_res)
            count += 1
            if count == MAX_RES_COUNT:
                log.warn(f"Showing first {MAX_RES_COUNT} cloudprofiles only.")
                break
        except config_not_present as e:
            log.warn(f"get cloudprofile failed for {binding.name!r}: {str(e)}")
            errno = NS_BASE_ERRORS['NSERR_NOENT']['errno']
            if errmsg != 'None':
                errmsg = f"{errmsg};{str(e)}"
            else:
                errmsg = str(e)

        if count and errno != '0':
            log.warn("Some cloudprofiles have configuration problems, please refer above warnings.")
            errno = 0
            errmsg = "None"

    if not count and errno == '0':
        if profile_name:
            errno = NS_BASE_ERRORS['NSERR_NOENT']['errno']
            errmsg = NS_BASE_ERRORS['NSERR_NOENT']['desc']
        else:
            errno = NS_BASE_ERRORS['NSERR_NODEV']['errno']
            errmsg = NS_BASE_ERRORS['NSERR_NODEV']['desc']
        print(json.dumps({'errno': errno, 'errmsg': errmsg, "response": []}))


def get_autoscalegroups(expected_grpname=None):
    result = []
    errno = NS_BASE_ERRORS['NSERR_NODEV']['errno']
    errmsg = NS_BASE_ERRORS['NSERR_NODEV']['desc']
    count = 0

    res, group_name, _ = _resolve_g_name(expected_grpname)
    if res:
        print(json.dumps(res))
        return

    try:
        groups = cloud.get_group_info(group_name)
    except (aws_authetication_failure, azure_authentication_failure) as e:
        errno = NS_BASE_ERRORS['NSERR_CLOUD_CRED']['errno']
        desc = NS_BASE_ERRORS['NSERR_CLOUD_CRED']['desc']
        errmsg = ' ' .join(str(e).split()[1:])
        result = [desc]
    except Exception as e:
        errno = NS_BASE_ERRORS['NSERR_SVC_GRP']['errno']
        desc = NS_BASE_ERRORS['NSERR_SVC_GRP']['desc']
        errmsg = str(e)
        result = [desc]
    else:
        configured_group = rain.group()

        for group in groups:
            try:
                configured_group = configured_group.get(config, group.name)
                # Configured groups are already drainable
                if cloud.get_cloud_platform() == 'AWS':
                    group.drain = True
            except config_not_present:
                pass

            asg_ordered = OrderedDict()
            for enum in range(len(CONST.PE_ORDER_GET_ASG)):
                if enum not in CONST.PE_ORDER_GET_ASG:
                    raise rainman_exception(
                        "index missing from dict keys, please check in get_autoscalegroups.")
                asg_ordered[CONST.PE_ORDER_GET_ASG[enum]] = group.__dict__[
                    CONST.PE_ORDER_GET_ASG[enum]]
            str_res = json.dumps({'errno': 0, 'errmsg': "None", "response": [asg_ordered]})
            str_res = str_res.replace("\n", " ")
            if len(str_res) > MAX_RES_LIST_LEN:
                log.warn(f"Skipping autoscale group {group.name!r} from display as output is too long.")
                continue
            print(str_res)
            count += 1
            if count == MAX_RES_COUNT:
                log.warn(f"Showing first {MAX_RES_COUNT} autoscalegroups only.")
                break

    if not count:
        print(json.dumps({'errno': errno, 'errmsg': errmsg, "response": result}))


def get_cloudips():
    client_intf_mac = None
    if cloud.get_cloud_platform() in ('AZURE', 'AZURESTACK'):
        nodestate = local.get_node_config()
        client_intf = "1/1"
        if nodestate in ['CCO', 'CL Node']:
            client_intf = "0/1/1"
        client_intf_mac = local.get_mac(client_intf)
        client_intf_mac = client_intf_mac.replace(':', '')

    cloudips = cloud.get_instance_freeips(client_intf_mac)

    count = 0
    result = []
    if isinstance(cloudips, str):
        ip_dict = {"cloudip": cloudips}
        result.append(ip_dict)
    elif cloudips is not None:
        for cloudip in cloudips:
            ip_dict = {"cloudip": cloudip}
            result.append(ip_dict)

    for ip_dict in result:
        str_res = json.dumps({'errno': 0, 'errmsg': "None", "response": [ip_dict]})
        str_res = str_res.replace("\n", " ")
        if len(str_res) >= MAX_RES_LIST_LEN:
            log.warn(f"Skipping cloudip {cloudip!r} from display as output is too long.")
            continue
        print(str_res)
        count += 1
        if count == MAX_RES_COUNT:
            log.warn(f"Showing first {MAX_RES_COUNT} cloudips only.")
            break

    if not count:
        errno = NS_BASE_ERRORS['NSERR_NODEV']['errno']
        errmsg = NS_BASE_ERRORS['NSERR_NODEV']['desc']
        print(json.dumps({'errno': errno, 'errmsg': errmsg, "response": []}))


def is_rain_configured():
    result = 0
    check_list = ['profile', 'lb', 'group']
    errmsg = None
    bindings = []
    lbs = []
    groups = []
    min_conf = 0

    for component in check_list:
        try:
            if component == 'profile':
                bindings = rain.group_lb_alarm_binding().get(config)
                if len(bindings) > 0:
                    log.debug(
                        "is_rain_configured : lb bindings is present in rain config")
                    return 1
            elif component == 'lb':
                lbs = rain.loadbalancer().get(config)
                if len(lbs) > 0:
                    log.debug(
                        "is_rain_configured : lb is present in rain config")
                    return 1
            elif component == 'group':
                groups = rain.group().get(config)
                if len(groups) > 0:
                    log.debug(
                        "is_rain_configured : autoscale group is present in rain config")
                    return 1
        except config_not_present as e:
            log.debug("is_rain_configured " + str(e))
            result = 0

    if cloud.get_cloud_platform() in ('AZURE', 'AZURESTACK', 'GCP'):
        min_conf = 1

    if local.get_lbs_in_ns() > min_conf or local.get_services_in_ns() > min_conf or local.get_sgs_in_ns():
        log.debug(
            "is_rain_configured : service/servicegroup/vservers configured in NS")
        result = 1

    return result


def get_ftumode(ftumode_file):
    groups = []
    ftumode = 0

    nodestate = local.get_node_config()
    if nodestate in ['CCO', 'CL Node']:
        set_ftumode(ftumode_file, 0)
        return 0
    try:
        with open(ftumode_file, "r") as fp:
            ftumode = fp.readline()
            fp.close()
    except IOError:
        intf_count = local.get_intf_count()
        if cloud.validate_intf_count(intf_count):
            log.debug("get_ftumode: Less than 3 interface : Disabling ftumode")
            ftumode = 0
        else:
            if cloud.get_cloud_platform() == 'AWS':
                qc = cloud.quick_api_call(cloud.get_group)
                groups = qc()
                is_conf_not_pre = is_rain_configured()
                if len(groups) > 0:
                    if is_conf_not_pre == 1:
                        log.debug(
                            "get_ftumode : lbvserver/service/servicegroup present, Setting FTU mode to 0 ")
                        ftumode = 0
                    else:
                        log.debug(
                            "get_ftumode : lbvserver/service/servicegroup not present, Setting FTU mode to 1 ")
                        ftumode = 1
            elif cloud.get_cloud_platform() in ('AZURE', 'AZURESTACK', 'GCP'):
                is_conf_not_pre = is_rain_configured()
                if is_conf_not_pre == 1:
                    log.debug(
                        "get_ftumode : lbvserver/service/servicegroup present, Setting FTU mode to 0 ")
                    ftumode = 0
                else:
                    log.debug(
                        "get_ftumode : lbvserver/service/servicegroup not present, Setting FTU mode to 1 ")
                    ftumode = 1
        try:
            with open(ftumode_file, "w") as fp:
                fp.write(str(ftumode))
                fp.close()
        except IOError as e:
            log.warn(
                f"get_ftumode: Not able to write to file {ftumode_file!r} , exception seen: {e}")
    return ftumode


def set_ftumode(ftumode_file, ftumode):
    '''
    FTU mode is decided based on Backend ASG and rainman
    configuration. Setting FTU mode to 1 from CLI/GUI
    is not supported
    '''
    if ftumode == '1':
        log.error("Manually enabling FTU mode is not supported.")
        return
    try:
        with open(ftumode_file, "w") as fp:
            fp.write(ftumode)
            fp.close()
    except IOError as e:
        log.debug("set_ftumode: Not able to write to file %s error:" +
                  str(e) % (ftumode_file))


def check_ftumode(ftumode_file):

    is_conf_not_pre = is_rain_configured()
    if is_conf_not_pre == 1:
        log.debug("check_ftumode : Resetting FTU mode ")
        try:
            os.remove(ftumode_file)
        except:
            pass


def argv_to_args(arguments, function):
    '''
    treats the arguments passed from the cli like:
    -name1 value1 -name2 value2

    And converts them into function arguments like:
    func(name1=value1, ...)

    '''

    # Fetch the valid arugments for the function we want to call with user supplied arguments
    func_args_tuple = inspect.getargspec(function)
    for func_arg in func_args_tuple:
        if type(func_arg) is list:
            allowed_func_args = func_arg

    user_args = arguments[3:]

    out_func_args = {}

    i = 0
    while i < len(user_args):
        if user_args[i].startswith("-"):
            if user_args[i].lstrip("-") in allowed_func_args:
                try:
                    out_func_args[user_args[i].lstrip("-")] = user_args[i+1]
                except IndexError:
                    raise config_args_invalid(
                        "%s does not have a value" % (user_args[i]))
            else:
                raise config_args_invalid(
                    "%s is not a supported argument for this command" % (user_args[i]))
        i = i+1
    print(str(out_func_args))
    return out_func_args


def signal_rain_scale(signo):
    rs_pid = 0
    rs_pid_file = cloud.get_daemon_pid_file()
    try:
        with open(rs_pid_file, "r") as fp:
            rs_pid = int(fp.read())
            os.kill(rs_pid, signo)
    except IOError:
        log.info('Not able to get rain_scale daemon process id')
    except OSError:
        log.info("Rain scale process not found with pid %d" % rs_pid)
    except ValueError as e:
        log.info("%s" % (str(e)))


def signal_rain_tags(signo):
    rs_pid = 0
    rs_pid_file = cloud.get_rain_tags_daemon_pid_file()
    try:
        with open(rs_pid_file, "r") as fp:
            rs_pid = int(fp.read())
            os.kill(rs_pid, signo)
            log.info("Signal %d sent to Rain tags process " % signo)
    except IOError:
        log.info('Not able to get rain_tags daemon process id')
    except OSError:
        log.info("Rain tags process not found with pid %d" % rs_pid)


def init_event_queue():
    queue_conf = rain.event_queue()
    try:
        queue_conf = rain.event_queue().get(config, "default")
    except (config_not_present, config_file_error):
        log.info("Adding queue_conf to rainman.conf...")
        queue_conf.name = "default"
        queue_conf.details = cloud.get_event_queue_details()
        if queue_conf.details is not None:
            cloud.add_event_queue(queue_conf)
            queue_conf.add(config, queue_conf)
            signal_rain_scale(signal.SIGUSR2)
    else:
        cloud.add_event_queue(queue_conf)
    return queue_conf


def get_event_queue():
    queue_conf = rain.event_queue()
    try:
        queue_conf = rain.event_queue().get(config, "default")
    except (config_not_present, config_file_error):
        log.info("Could not find any queue_conf. Returning None ...")
        return None
    return queue_conf


def create_event_queue():
    try:
        groups = rain.group().get(config)
    except config_not_present:
        log.debug("groups not configured")
        return None
    else:
        queue_conf = init_event_queue()
        if not queue_conf:
            log.warn("group found but unable to configure event queue.")
        return queue_conf


def remove_event_queue():
    queue_conf = rain.event_queue()
    try:
        queue_conf = rain.event_queue().get(config, "default")
    except (config_not_present, config_file_error):
        log.warn("removing event queue failed")
    else:
        cloud.remove_event_queue(queue_conf)
        queue_conf.remove(config, queue_conf)
        log.debug("successfully removed event queue")


def delete_event_queue():
    delete_queue = False
    autoscale_group_found = False
    try:
        groups = rain.group().get(config)
    except config_not_present:
        delete_queue = True
    else:
        for group in groups:
            if group.cloud_profile_type == 1:
                autoscale_group_found = True

        if autoscale_group_found == False:
            delete_queue = True

    if delete_queue == True:
        remove_event_queue()


def checkiam_autoscale():
    print("RAINMAN CHECIAM AUTOSCALE ")


def monitor_autoscale(expected_grpname, proto,  port, drain, delay):
    res, grpname, servicename = _resolve_g_sg_names(expected_grpname, port)
    if res:
        print(json.dumps(res))
        return

    # backward compatibility scenario
    if expected_grpname == grpname:
        log.info(
            "servicegroup name is same as autoscale, we are in backward compatible scenario.")
        result = check_sg_not_configured(grpname)
        if result:
            print(json.dumps(result))
            return

    result = check_sg_not_configured(servicename)
    if result:
        print(json.dumps(result))
        return

    result, zone, rg = check_cloud_asg_exist(grpname)
    if result:
        print(json.dumps(result))
        return

    try:
        add_group(grpname, expected_grpname, proto, port, drain, delay,
                  'SERVICEGROUP', CONST.AUTOSCALE, zone=zone, rg=rg)
        if expected_grpname == grpname:
            # create dummy entry as per servicename convention for port
            add_service(grpname, servicename, port,
                        proto, CONST.AUTOSCALE, enabled=False)
        log.info("monitor autoscale: servicegroup added.")
    except:
        rlog.error_trace("exception seen while adding servicegroup.")
        remove_autoscale(expected_grpname)

    print(json.dumps({'response': 'success',
                      'errno': 0, 'errmsg': "servicegroup added."}))
    if cloud.get_cloud_platform() == 'GCP':
        try:
            signal_rain_scale(signal.SIGUSR2)
        except:
            log.warn("failed to signal rain_scale.")


def set_servicegroup(expected_grpname, drain, delay):
    res, grpname, _ = _resolve_g_name(expected_grpname)
    if res:
        print(json.dumps(res))
        return

    result = check_g_sg_configured(grpname, expected_grpname)
    if result:
        print(json.dumps(result))
        return

    try:
        group = rain.group().get(config, grpname)
    except config_not_present as e:
        errno = str(e).split()[0].rstrip(',')
        errmsg = ' ' .join(str(e).split()[1:])
        print(json.dumps({'response': 'failed',
                          'errno': errno, 'errmsg': errmsg}))
        return
    else:
        log.info("Setting up drain %s and delay %s for group %s" %
                 (drain, delay, grpname))
        group.drain = drain
        group.drain_time = delay
        group.update(config, group)

    print(json.dumps({'response': 'success', 'errno': 0,
                      'errmsg': "servicegroup updated."}))
    return


def remove_autoscale(expected_grpname):
    res, grpname, servicename = _resolve_g_name(expected_grpname)
    if res:
        print(json.dumps(res))
        return

    try:
        if expected_grpname == grpname:
            _remove_service(grpname)
        _remove_service(servicename)
        _remove_group(grpname)
    except:
        rlog.error_trace("exception seen while removing autoscale, ignoring.")

    print(json.dumps({'response': 'success', 'errno': 0,
                      'errmsg': "servicegroup removed."}))


def clear_autoscale(expected_servicename):
    log.info(f"clear autoscale {expected_servicename!r}")
    res, grpname, servicename = _resolve_g_name(expected_servicename)
    if res:
        print(json.dumps(res))
        return

    if expected_servicename != servicename:
        print(json.dumps({'response': 'success', 'errno': 0,
                                  'errmsg': f"invalid servicegroup name provided. rainman ignored clear command."}))
        return

    try:
        service = rain.service().get(config, servicename)
        if service:
            # block servicegroup cleanup if servicegroup is part of present bindings/CP
            try:
                bindings = rain.group_lb_alarm_binding().get(config)
                for binding in bindings:
                    if binding.service == servicename:
                        print(json.dumps({'response': 'failed', 'errno': NS_BASE_ERRORS['NSERR_PERM']['errno'],
                                  'errmsg': f"servicegroup {servicename} is part of a cloud profile. Operation not permitted"}))
                        return
            except config_not_present:
                pass
            log.info(f"cleaning servicegroup {service.name!r}")
            server_list = local.get_servers_in_group(service)
            for server in server_list:
                try:
                    local.remove_server_from_group(server, service)
                except:
                    rlog.error_trace("Exception seen while cleaning servicegroup")
    except:
        rlog.error_trace()

    log.info(f'Group {grpname!r} is cleared.')
    print(json.dumps({'response': 'success', 'errno': 0,
                      'errmsg': "servicegroup cleaned."}))


def prepe_validate(expected_grpname, port, proto):
    res, grpname, servicename = _resolve_g_sg_names(expected_grpname, port)
    if res:
        print(json.dumps(res))
        return

    log.info("Checking servicegroup")
    result = check_sg_not_configured(expected_grpname)
    if result:
        print(json.dumps(result))
        return

    result = check_sg_not_configured(servicename)
    if result:
        print(json.dumps(result))
        return

    log.info("Checking autoscalegroup")
    result, zone, rg = check_cloud_asg_exist(grpname)
    if result:
        print(json.dumps(result))
        return

    log.info('Prepe validation passed.')
    print(json.dumps({'response': 'success', 'errno': 0,
                      'errmsg': "Prepe validation passed."}))


def main():
    '''
    Usage of rain_config mimics NSCLI

    ex.
            rain_config show autoscalegroups
            rain_config show cloudprofile
            rain_config show cloudips
            rain_config add cloudprofile ...
            rain_config checkiam autoscale ...
            rain_config monitor autoscale ...
    '''

    primary = False
    cmds = ["show", "add", "remove", "set", "unset", "checkiam",
            "monitor", "clear", "check", "prepe_validate"]
    cmd_groups = ["alarm", "cloudprofile", "autoscalegroups",
                  "cloudips", "ftumode", "autoscale", "servicegroup", "privileges"]
    local_cmd_groups = ["ftumode"]
    ftumode_file = cloud.get_ftu_filename()

    log.info(f"process called, args: {sys.argv!r}")

    # need at least two arguments
    if (len(sys.argv) < 3):
        log.error("Wrong number of arguments provided = %s" % sys.argv)
        print("Required argument missing.")
        return -1

    user_cmd = sys.argv[1]
    user_cmd_group = sys.argv[2]

    if not user_cmd in cmds or not user_cmd_group in cmd_groups:
        log.error("Unsupported arguments provided = %s" % sys.argv)
        print("Unsupported operation.")
        return -1

    f = None
    pre_stdout = None
    output = ""
    ipc_conf_lock = get_conf_lock()

    try:
        ipc_conf_lock.acquire()
        pre_stdout = sys.stdout
        f = open(cloud.log_folder+user_cmd+"_"+user_cmd_group+".json", "w")
        sys.stdout = f

        # set/unset servicegroup commands are considered as local command
        if not (user_cmd_group in ["servicegroup", "privileges", "ftumode"] and user_cmd in ["set", "unset", "check"]):
            primary = is_primary_node()
            if primary != True and user_cmd != "show":
                log.debug("Not on a primary node. %d exiting..." % os.getpid())
                output = (json.dumps({'response': 'done', 'errno': 0,
                                      'errmsg': "Rainman command is not supported in Seconary/non-CCO node"}))
                print(output)
                return 0

        if cloud.is_authenticated() == False:
            if not user_cmd_group in local_cmd_groups:
                errno = NS_BASE_ERRORS['NSERR_CLOUD_CRED']['errno']
                desc = NS_BASE_ERRORS['NSERR_CLOUD_CRED']['desc']
                result = {'response': desc, 'errno': errno,
                          'errmsg': 'Cloud credentials not set'}
                output = (json.dumps(result))
                print(output)
                return 0

        if user_cmd == "add":
            if user_cmd_group == "cloudprofile":
                node_type = local.get_node_config()
                if node_type in ['CCO', 'CL Node']:
                    errno = NS_BASE_ERRORS['NSERR_NOT_SUPPINCLUSTER']['errno']
                    desc = NS_BASE_ERRORS['NSERR_NOT_SUPPINCLUSTER']['desc']
                    result = {'response': desc, 'errno': errno,
                              'errmsg': 'Cloud profile configuration is not supported in cluster'}
                    output = (json.dumps(result))
                    print(output)
                    return 0
                if (len(sys.argv) < 14):
                    print("Wrong number of arguments.")
                    print(
                        f"Usage: {sys.argv[0]} add cloudprofile <profile_name> <profile_type> <lbserver_name> <Asg_name> <vsrvr_proto> <IP> <port>  <svcProto> <Svc_port> < Drain> <delay>")
                    return -1
                ## add_cloudprofile("ASG_PROFILE", "ASG_LINUX_MICRO_ASG_LB", ["ASG_LINUX_MICRO_ASG"], "http", "1.1.1.1", 80, "http", 80, True, 300)

                profile_type = sys.argv[4]
                vsrvr_proto_code = sys.argv[7]
                svcProto_code = sys.argv[10]

                if len(vsrvr_proto_code) == 1:
                    vsrvr_proto_code = '0' + vsrvr_proto_code

                if vsrvr_proto_code not in list(RAIN_PROTOCOLS.keys()):
                    errno = NS_BASE_ERRORS['NSERR_INVAL']['errno']
                    desc = NS_BASE_ERRORS['NSERR_INVAL']['desc']
                    errmsg = "Invalid protocol code %s." % sys.argv[7]
                    print(json.dumps(
                        {'response': desc, 'errno': errno, 'errmsg': errmsg}))
                    return -1
                else:
                    vsrvr_proto = RAIN_PROTOCOLS[vsrvr_proto_code]

                if len(svcProto_code) == 1:
                    svcProto_code = '0' + svcProto_code

                if svcProto_code not in list(RAIN_PROTOCOLS.keys()):
                    errno = NS_BASE_ERRORS['NSERR_INVAL']['errno']
                    desc = NS_BASE_ERRORS['NSERR_INVAL']['desc']
                    errmsg = "Invalid protocol code %s." % sys.argv[10]
                    print(json.dumps(
                        {'response': desc, 'errno': errno, 'errmsg': errmsg}))
                    return -1
                else:
                    svcProto = RAIN_PROTOCOLS[svcProto_code]

                log.debug("add cloudprofile: input args checks done.")
                if (int(profile_type) == int(CONST.AUTOSCALE)):
                    add_cloudprofile(sys.argv[3], sys.argv[5], sys.argv[6], vsrvr_proto, sys.argv[8],
                                     sys.argv[9], svcProto, sys.argv[11], sys.argv[12], sys.argv[13])
                elif (int(profile_type) == int(CONST.AZURETAGS)):
                    add_cloudprofile_by_tag(sys.argv[3], sys.argv[5], sys.argv[6], vsrvr_proto, sys.argv[8],
                                            sys.argv[9], svcProto, sys.argv[11], sys.argv[12], sys.argv[13], sys.argv[14])
                if get_ftumode(ftumode_file) == '1':
                    set_ftumode(ftumode_file, '0')

        elif user_cmd == "remove":
            if user_cmd_group == "cloudprofile":
                node_type = local.get_node_config()
                if node_type in ['CCO', 'CL Node']:
                    errno = NS_BASE_ERRORS['NSERR_NOT_SUPPINCLUSTER']['errno']
                    desc = NS_BASE_ERRORS['NSERR_NOT_SUPPINCLUSTER']['desc']
                    result = {'response': desc, 'errno': errno,
                              'errmsg': 'Cloud profile configuration is not supported in cluster'}
                    print(json.dumps(result))
                    return 0
                if (len(sys.argv) < 4):
                    print("Wrong number of arguments.")
                    print("Usage: " + sys.argv[0] +
                          " remove cloudprofile <profile_name>")
                    return -1
                remove_cloudprofile(sys.argv[3])
            elif user_cmd_group == "servicegroup":
                if (len(sys.argv) < 4):
                    print("Wrong number of arguments.")
                    print(
                        f"Usage: {sys.argv[0]} remove servicegroup <servicegroup_name>")
                    return -1
                remove_autoscale(sys.argv[3])
        elif user_cmd == "show":
            if user_cmd_group == "cloudprofile":
                profile_name = None
                if (len(sys.argv) == 4):
                    profile_name = sys.argv[3]
                get_cloudprofile(profile_name)
            if user_cmd_group == "autoscalegroups":
                group_name = None
                if (len(sys.argv) == 4):
                    group_name = sys.argv[3]
                get_autoscalegroups(group_name)
            elif user_cmd_group == "cloudips":
                get_cloudips()
            elif user_cmd_group == "ftumode":
                get_ftumode(ftumode_file)
        elif user_cmd == "set":
            if user_cmd_group == "ftumode":
                if (len(sys.argv) == 4):
                    set_ftumode(ftumode_file, sys.argv[3])
            if user_cmd_group == "servicegroup":
                if (len(sys.argv) == 6):
                    set_servicegroup(sys.argv[3], sys.argv[4], sys.argv[5])
        elif user_cmd == "unset":
            if user_cmd_group == "servicegroup":
                if (len(sys.argv) == 6):
                    set_servicegroup(sys.argv[3], "0", "false")
        elif user_cmd == "checkiam":
            if user_cmd_group == "autoscale":
                checkiam_autoscale()
        elif user_cmd == "check":
            if user_cmd_group == "privileges":
                if len(sys.argv) < 4:
                    print("Error: Wrong number of arguement! Feature not specified!")
                    return -1
                return cloud.check_privileges(sys.argv[3])

        elif user_cmd == "prepe_validate":
            if user_cmd_group == "autoscale":
                if (len(sys.argv) < 6):
                    print("Wrong number of arguments.")
                    print(
                        "Usage: " + sys.argv[0] + " prepe_validate autoscale <Asg_name> <svcProto> <Svc_port> < Drain> <delay>")
                    return -1
                ## monitor_autoscale(["ASG_LINUX_MICRO_ASG"], "http", 80, True, 300)

                svcProto_code = sys.argv[4]
                if len(svcProto_code) == 1:
                    svcProto_code = '0' + svcProto_code

                if svcProto_code not in list(RAIN_PROTOCOLS.keys()):
                    errno = NS_BASE_ERRORS['NSERR_INVAL']['errno']
                    desc = NS_BASE_ERRORS['NSERR_INVAL']['desc']
                    errmsg = "Invalid protocol code %s." % sys.argv[4]
                    print(json.dumps(
                        {'response': desc, 'errno': errno, 'errmsg': errmsg}))
                    return -1
                if sys.argv[5] == "0":
                    errno = NS_LB2_ERRORS['NSERR_MEMBERPORT_REQD']['errno']
                    desc = NS_LB2_ERRORS['NSERR_MEMBERPORT_REQD']['desc']
                    errmsg = "Invalid port number %s." % sys.argv[5]
                    print(json.dumps(
                        {'response': desc, 'errno': errno, 'errmsg': errmsg}))
                    return -1

                prepe_validate(sys.argv[3], sys.argv[5], svcProto_code)

        elif user_cmd == "monitor":
            if user_cmd_group == "autoscale":
                # Usage: monitor autoscale <Asg_name> <svcProto> <Svc_port> < Drain> <delay>
                svcProto_code = sys.argv[4]
                if len(svcProto_code) == 1:
                    svcProto_code = '0' + svcProto_code
                monitor_autoscale(
                    sys.argv[3], svcProto_code, sys.argv[5], sys.argv[6], sys.argv[7])
        elif user_cmd == "clear":
            if user_cmd_group == "servicegroup":
                if (len(sys.argv) < 4):
                    print("Wrong number of arguments.")
                    print(
                        "Usage: " + sys.argv[0] + " clear servicegroup <servicegroup_name>")
                    return -1
                clear_autoscale(sys.argv[3])
        else:
            print("Unsupported command arguments.")
            # dont reach
            return -1
    except (config_args_invalid, TypeError) as e:
        print("Incorrect command arguments, %s" % (str(e)))
    finally:
        ipc_conf_lock.release()
        if f:
            f.close()
        if pre_stdout:
            sys.stdout = pre_stdout
    return 0


def is_primary_node():
    nodestate = local.get_node_config()
    if nodestate in ['Primary', 'CCO', 'StandAlone']:
        return True
    return False


if __name__ == "__main__":
    try:
        sys.exit(main())
    except:
        if log:
            rlog.error_trace()
        sys.exit(-1)
