#!/usr/bin/env python
"""
Copyright 2000-2022 Citrix Systems, Inc. All rights reserved.
This software and documentation contain valuable trade secrets
and proprietary property belonging to Citrix Systems, Inc.
None of this software and documentation may be copied,
duplicated or disclosed without the express written permission
of Citrix Systems, Inc.
"""


import sys
import time
import datetime
import json
import os
import psutil
import boto3
import re
import base64
import traceback

from azurelinuxagent.common.osutil.nsvpx import NSVPXOSUtil

import rainman_core.common.constants as CONST
from rainman_core.common.logger import RainLogger
from rainman_core.common import rain
from rainman_core.common.exception import *
from rainman_core.common.stats import stats_config

rlog = RainLogger(CONST.STATS_LOG_FILE_NAME, CONST.DEFAULT_LOG_LEVEL)
log = rlog.logger


class Rain_stats(object):
    def __init__(self):
        self.config = rain.rainman_config()
        self.local = self.config.get_local_config_service()
        self.cloud = self.config.get_cloud_config_service()
        self.reporting = self.config.get_cloud_reporting_service()
        self.pid_file = self.cloud.get_rain_stats_daemon_pid_file()

        self.config_reload_period_sec = 30
        self.sec_count = 0
        self.namespace = "ADC"
        self.do_publish_metrics = "default"
        self.main_loop_period_sec = 0
        self.publish_interval = 0
        self.regex_enabled = re.compile('^(yes|true|on|enabled|enable)$', re.IGNORECASE)
        self.regex_disabled = re.compile('^(no|false|off|disabled|disable)$', re.IGNORECASE)
        self.regex_default = re.compile('^(UserData|user_data|default)$', re.IGNORECASE)
        self.stats = stats_config()
        self.platform = self.cloud.get_cloud_platform()

    def is_primary_node(self):
        nodestate = self.local.get_node_config()
        if nodestate in ['Secondary', 'NON-CCO']:
            return False
        return True

    def check_running_process(self):
        ret = 0

        for proc in psutil.process_iter():
            try:
                pinfo = proc.as_dict(attrs=['pid', 'cmdline'])
                cmdline = pinfo['cmdline']
                if len(cmdline) > 1 and 'python' in cmdline[0] and 'rain_stats' in cmdline[1] and pinfo['pid'] != os.getpid():
                    log.warning("Another process %d is running, %d exiting...", pinfo['pid'], os.getpid())
                    ret = 1
            except psutil.NoSuchProcess:
                pass
        return ret

    def clear_pid_file(self):
        try:
            os.remove(self.pid_file)
        except OSError:
            pass

    def exit_daemon(self, ret_code):
        self.clear_pid_file()
        exit(ret_code)

    def get_config_from_file(self, file_path, data_regex='^([^=]+?)=(.*?)$', name_group_index=1, value_group_index=2, strip=True):
        cfg = {}
        try:
            cfgfile = open(file_path, "r")
            pat = re.compile(data_regex)
            for line in cfgfile:
                m = pat.match(line)
                if m:
                    cfg.update(dict([[x.strip() if strip else x for x in m.group(name_group_index, value_group_index)]]))
        except Exception as e:
            log.warning("Unable to extract configuration from file %s (%s)", file_path, e)
        return cfg

    def get_user_data_aws(self, param_name_re=None):
        if param_name_re is None:
            param_name_re = '[^=]+?'
        regex = '^({})=(.*?)$'.format(param_name_re)
        user_data = self.get_config_from_file(file_path='/flash/nsconfig/.AWS/nws_details', data_regex=regex, name_group_index=1, value_group_index=2, strip=True)
        return user_data.get(param_name_re, '')

    def get_user_data_azure(self, param_name_re=None):
        try:
            with open('/flash/nsconfig/.AZURE/customdata', 'r') as f_b:
                encoded_data = f_b.read()
            decoded_data = base64.b64decode(encoded_data)
            custom_data = json.loads(decoded_data)
            return custom_data.get("vpx_config", '').get(param_name_re, None)
        except Exception as err:
            log.warning("Exception while fetching configuration from custom data config file. (%s)", str(err))
            return None

    def get_user_data(self):
        result = {}
        platform = self.cloud.get_cloud_platform()
        if platform == "AWS":
            result = self.get_user_data_aws('PublishCloudwatchMetrics')
        if platform == "AZURE":
            result = self.get_user_data_azure('PublishMonitoringMetrics')
        return result

    def do_publish(self, do_publish_metrics):
        if self.regex_disabled.match(do_publish_metrics):
            # If do_publish_metrics in stats json is no/off/disabled skip publishing metrics
            log.debug("Skipping publishing metrics (do_publish_metrics = %s)", do_publish_metrics)
            return False
        elif NSVPXOSUtil.check_ns_on_azstack():
            log.debug("Custom metrics are not supported in Azure Stack Hub")
            return False
        elif self.regex_enabled.match(do_publish_metrics):
            # If do_publish_metrics in stats json is yes/on/enabled proceed publishing metrics
            log.debug("Proceed publishing metrics (do_publish_metrics = %s)", do_publish_metrics)
            return True
        elif self.regex_default.match(do_publish_metrics):
            # If do_publish_metrics in stats json has a value of "default" or "user_data" then check user_data...
            publish_metrics_ud = self.get_user_data()
            if  publish_metrics_ud is not None and self.regex_disabled.match(publish_metrics_ud):
                # if PublishCloudwatchMetrics or PublishMonitoringMetrics in user_data is no/off/disabled do not publish metrics
                log.debug("Skip publishing metrics according to Custom/User Data (Publish Metrics = %s)", publish_metrics_ud)
                return False
            elif publish_metrics_ud is None and self.cloud.get_cloud_platform() == "AZURE":
                # if no PublishMonitoringMetrics in user_data and environment is Azure do not publish metrics
                return False
            else:
                # By default do publish metrics
                log.debug("Proceed publishing metrics according to Custom/User Data (Publish Metrics = %s)", publish_metrics_ud)
                return True
        else:
            # There is no do_publish_metrics in stats json. Do publish metrics by default.
            log.debug("Proceed publishing metrics (do_publish_metrics does not exist)")
            return True

    def reload_config(self):
        try:
            self.stats.reload_config()
        except (ValueError, IOError) as e:
            log.error("Cannot load rain_stats config (existing/default config will be used): %s", e)
        else:
            metrics = self.stats.config.get('metrics', [])
            self.namespace = self.stats.config.get('namespace', 'Citrix ADC')
            loglevel = self.stats.config.get('loglevel', 'INFO')
            try:
                log.setLevel(loglevel)
                log.debug("Setting log level to \"%s\"", loglevel)
            except e:
                log.error("Invalid log level \"%s\"", loglevel)
            # Get new sampling period (minimum valid value=7, default 60)
            new_sampling_period_sec = max([int(self.stats.config.get("sampling_period", 60)), 7])
            # Get new publish period (minimum valid value is sampling_period_sec, default 60)
            new_publish_period_sec = max([int(self.stats.config.get("publish_period", 60)), new_sampling_period_sec])
            # Get do_publish_metrics top configuration in json (default value is "default")
            self.do_publish_metrics = self.stats.config.get("do_publish_metrics", "default")
            if new_publish_period_sec != self.publish_period_sec or new_sampling_period_sec != self.sampling_period_sec:
                # If there is a change in sampling or publishing period then asign new values and recalculate main_loop_period_sec
                self.publish_period_sec = new_publish_period_sec
                self.sampling_period_sec = new_sampling_period_sec
                self.main_loop_period_sec = calculate_greater_common_divisor([self.publish_period_sec, self.sampling_period_sec])
                self.sec_count = 0
                self.publish_interval = time.time()
                log.debug("Set publish_period_sec=%d and sampling_period_sec=%d", self.publish_period_sec, self.sampling_period_sec)
                log.debug("Applying new loop period %s", self.main_loop_period_sec)
        return metrics

    def run(self):

        if self.is_primary_node() != True:
            log.info("Not on a primary node. %d exiting...", os.getpid())
            exit(9)
        if self.check_running_process() != 0:
            exit(10)
        log.info("rain_stats process %d starting...", os.getpid())

        try:
            with open(self.pid_file, "w") as fp:
                fp.write("%d" % os.getpid())
        except IOError as e:
            log.error("Not able to create pid file %s error: %s", self.pid_file, str(e))

        # Get new sampling period (minimum valid value=7, default 60)
        self.sampling_period_sec = max([int(self.stats.config.get("sampling_period", 60)), 7])
        # Get new publish period (minimum valid value is sampling_period_sec, default 60)
        self.publish_period_sec = max([int(self.stats.config.get("publish_period", 60)), self.sampling_period_sec])
        log.debug("Set publish_period_sec=%d and sampling_period_sec=%d", self.publish_period_sec, self.sampling_period_sec)

        # Use GCD of the above periods as the main loop period
        self.main_loop_period_sec = calculate_greater_common_divisor([self.publish_period_sec, self.sampling_period_sec])
        last_config_load_ts = 0
        self.publish_interval = time.time()
        metric_data = []

        while 1:
            loop_start_ts = self.publish_interval
            if loop_start_ts - last_config_load_ts >= self.config_reload_period_sec:
                # load rain_stats.json configuration not earlier than every 30 seconds.
                last_config_load_ts = loop_start_ts
                metrics = self.reload_config()
            # Now execute sampling and publishing if needed...
            if self.do_publish(self.do_publish_metrics):
                if self.sec_count % self.sampling_period_sec == 0:
                    log.debug("Sampling metrics: sec_count=%d loop_start_ts=%.3f", self.sec_count, loop_start_ts)
                    for metric in metrics:
                        # nitro API call to get system statistics
                        resource = metric["resource"]
                        try:
                            system_stat = self.local.get_system_stat(resource)
                            # formulate metric data per cloud provider
                            instance_name_id = self.reporting.instanceName + " " + self.reporting.instanceid
                            log.debug("instance_name_id = '%s'", instance_name_id)
                            metric["dimensions"][0]["Value"] = instance_name_id
                            data = self.reporting.formulate_metric(metric, system_stat, self.stats)
                            metric_data.append(data)
                        except Exception as e:
                            log.error("Error geting system stat: %s", getattr(e, 'message', repr(e)))

                # publish metrics to cloud provider
                if self.sec_count % self.publish_period_sec == 0:
                    log.debug("Publishing metrics: sec_count=%d len(metrics)=%d len(metric_data)=%d loop_start_ts=%.3f", self.sec_count, len(metrics), len(metric_data), loop_start_ts)
                    if len(metrics) > 0:
                        try:
                            self.reporting.publish_metrics(metric_data, self.namespace)
                        except Exception as e:
                            log.error("Error publishing metrics: %s", getattr(e, 'message', repr(e)))
                    else:
                        log.warn("Rain stats metric list is empty. Check /nsconfig/rain_stats.json file")
                    metric_data = []

            # Sleeping until the next iteration
            time_until_next_iteration = self.main_loop_period_sec - (time.time() - loop_start_ts)
            if time_until_next_iteration <= 0:
                log.error("rain_stats main loop period (%.3f) too small! Consider increasing sampling perion.", self.main_loop_period_sec)
                time.sleep(1)
            else:
                log.debug("rain_stats main loop sleeping for %.4f seconds", time_until_next_iteration)
                time.sleep(time_until_next_iteration)
            self.sec_count += self.main_loop_period_sec
            self.publish_interval += self.main_loop_period_sec

def main(args=[]):
    """
    Parse command line arguments, exit with usage() on error.
    Invoke different methods according to different command
    """
    if len(args) > 0:
        print("Rain_stats binary does not accept any parameter. ")
    else:
        try:
            Rain_stats_obj = Rain_stats()
            Rain_stats_obj.run()
        except Exception:
            rlog.error_trace("Failed to run rain_stats proccess")

def calculate_greater_common_divisor(numbers):
    minimum = min(numbers)
    for i in range(minimum, 1, -1):
        is_common_divisor = True
        for n in numbers:
            if n % i != 0:
                is_common_divisor = False
                break
        if is_common_divisor:
            return i
    return 1

if __name__ == "__main__":
    main()
