#!/usr/bin/env python
"""
Copyright 2000-2022 Citrix Systems, Inc. All rights reserved.
This software and documentation contain valuable trade secrets
and proprietary property belonging to Citrix Systems, Inc.
None of this software and documentation may be copied,
duplicated or disclosed without the express written permission
of Citrix Systems, Inc.
"""


import sys
import time
import json
import os
import psutil
import signal

import rainman_core.common.constants as CONST
from rainman_core.common.logger import RainLogger
from rainman_core.common import rain
from rainman_core.common.exception import *

rlog = RainLogger(CONST.TAGS_LOG_FILE_NAME, CONST.DEFAULT_LOG_LEVEL)
log = rlog.logger

config = rain.rainman_config()
local = config.get_local_config_service()
cloud = config.get_cloud_config_service()

pollers = dict()
configured_groups = list()
pid_file = ""


def handle_signal(signum, frame):
    log.info("Received signal %d", signum)
    if signum not in (signal.SIGUSR1, signal.SIGUSR2, signal.SIGTERM, signal.SIGHUP):
        return

    if signum == signal.SIGUSR1:
        log.info("No longer primary/CCO node, exiting")
        exit_daemon(11)

    if signum == signal.SIGUSR2:
        log.info("Received SIGUSR2. Reconfiguring...")
        reconfigure()
        return

    if signum != signal.SIGTERM:
        return

    ha_state = local.get_ha_node_state()
    if ha_state == 'Secondary':
        log.info("SIGTERM signal received, No longer primary, Exiting...")
        exit_daemon(11)
    return


def get_configured_groups(cloud_profile_type=3):
    try:
        return [conf_group.name for conf_group in rain.group().get(config, None) if conf_group.cloud_profile_type is cloud_profile_type]
    except (config_not_present, config_file_error):
        log.warning("No configured groups found.")
        return []


def is_primary_node():
    nodestate = local.get_node_config()
    if nodestate in ['Primary', 'CCO', 'StandAlone']:
        return True
    return False


def wait_for_config():
    signal.pause()
    try:
        # return rain.azuretag().get(config, None)
        return get_configured_groups()
    except (config_not_present, config_file_error):
        log.info("No tags configured yet...")
        return None


def clean_wait_for_config(tk_refresh_th):
    # Stoping token refresh thread
    if tk_refresh_th is not None:
        tk_refresh_th.cancel()

    groups = wait_for_config()
    if groups is None:
        return None

    # starting token refresh thread
    if tk_refresh_th is not None:
        tk_refresh_th.start()
    return groups


def wait_for_event(tk_refresh_th):
    log.info("No tags configured, Sleeping")
    if cloud.get_cloud_platform() == "AZURE":
        return clean_wait_for_config(tk_refresh_th)
    log.error("Azure Tags daemon is not supported")
    exit(12)


def add_polling_for_group(group_name):
    log.debug("add_polling_for_group {}".format(group_name))
    pollers[group_name] = rain.aztags_poller(group_name)
    pollers[group_name].start_polling()


def remove_polling_for_group(group_name):
    p = pollers[group_name]
    p.stop_polling()
    del pollers[group_name]
    log.debug("Removed polling for group {}".format(group_name))


def configure_polling_for_groups(group_names):
    added_configured_groups = []
    removed_configured_groups = []

    for group_name in group_names:
        if group_name in configured_groups:
            pass
        else:
            added_configured_groups.append(group_name)

    for group_name in configured_groups:
        if group_name in group_names:
            pass
        else:
            removed_configured_groups.append(group_name)

    for group_name in removed_configured_groups:
        configured_groups.remove(group_name)
        remove_polling_for_group(group_name)

    for group_name in added_configured_groups:
        configured_groups.append(group_name)
        add_polling_for_group(group_name)


def reconfigure():
    configured_groups = get_configured_groups()

    try:
        configure_polling_for_groups(configured_groups)
    except config_failed as e:
        log.warning("Configuration failed: %s", str(e))


def main():

    log.debug("RainTags: Launching...")

    timer = None
    tk_refresh_th = None
    doit = 1
    pid_file = cloud.get_rain_tags_daemon_pid_file()

    try:
        with open(pid_file, "w") as fp:
            fp.write("%d" % os.getpid())
    except IOError as e:
        rlog.error_trace("Not able to create pid file %s", pid_file)

    signal.signal(signal.SIGTERM, handle_signal)
    signal.signal(signal.SIGUSR1, handle_signal)
    signal.signal(signal.SIGUSR2, handle_signal)

    reconfigure()

    log.debug("RainTags: Entering main loop...")
    previous = time.time() - 30
    while doit:

        current = time.time()
        time_diff = current - previous

        if time_diff >= 30:

            reconfigure()

            previous = current

        time.sleep(1)
    exit_daemon(0)


def check_running_process():
    ret = 0

    for proc in psutil.process_iter():
        try:
            pinfo = proc.as_dict(attrs=['pid', 'cmdline'])
            cmdline = pinfo['cmdline']
            if len(cmdline) > 1 and 'python' in cmdline[0] and 'rain_tags' in cmdline[1] and pinfo['pid'] != os.getpid():
                print("Another rain_tags process %d is running, %d exiting... - parent pid %d" %
                      (pinfo['pid'], os.getpid(), os.getppid()))
                ret = 1
        except psutil.NoSuchProcess:
            pass

    return ret


def clear_pid_file():
    try:
        os.remove(pid_file)
    except OSError:
        pass


def exit_daemon(ret_code):
    clear_pid_file()
    exit(ret_code)


ret = is_primary_node()
if ret != True:
    log.warning("Not on a primary node. %d exiting...", os.getpid())
    exit(9)

ret = check_running_process()
if ret != 0:
    print("Already running. %d exiting..." % os.getpid())
    exit(12)

if __name__ == "__main__":
    main()
