"""
Copyright 2000-2023 Citrix Systems, Inc. All rights reserved.
This software and documentation contain valuable trade secrets
and proprietary property belonging to Citrix Systems, Inc.
None of this software and documentation may be copied,
duplicated or disclosed without the express written permission
of Citrix Systems, Inc.
"""

from libnitrocli import nitro_cli, nitro_cli_output_parser

from rainman_core.common.base import base_local_driver
from rainman_core.common.rain import server
from rainman_core.common.exception import config_failed
from rainman_core.common.logger import RainLogger


class ns(base_local_driver):
    log = RainLogger.getLogger()
    info = log.info
    warn = log.warning
    debug = log.debug
    error = log.error

    def __init__(self, timeout=500):
        self.__nitro = nitro_cli(timeout=timeout)
        self.__parser = nitro_cli_output_parser()

    def _raise_on_fail(self, res):
        if not self.__parser.success(res):
            self.error("nitro cli failed with output '%s'", res)
            raise config_failed("nitro cli failed: %s" % res)

    def get_mac(self, name):
        out = self.__nitro.get_interface(name)
        if "interface" in out and len(out["interface"]) == 1:
            if "mac" in out["interface"][0]:
                return out["interface"][0]["mac"]
        self.warn(
            "failed to fetch mac for interface '%s'. nitro output: '%s'",
            name,
            out)
        raise config_failed(
            "failed to fetch mac for interface '%s'. nitro output: '%s'" %
            (name, out))

    def get_intf_count(self):
        return self.__nitro.get_interface_count()

    def add_group(self, group):
        self.info("Add new servicegroup: %s", group.name)
        sg_json = self.__nitro.get_servicegroup(group.name)
        if self.__parser.no_such_resource(sg_json):
            res = self.__nitro.add_servicegroup(group.name, group.protocol)
            self._raise_on_fail(res)
        else:
            self.warn("servicegroup '%s' already exists.", group.name)

    def remove_server(self, name):
        out = self.__nitro.get_servicegroup()
        if self.__parser.success(out) and 'servicegroup' in out:
            for sg_dict in out['servicegroup']:
                try:
                    out2 = self.__nitro.get_sg_sgm_b(
                        sg_dict['servicegroupname'])
                    if self.__parser.success(out2) and "servicegroup_servicegroupmember_binding" in out2:
                        members = out2["servicegroup_servicegroupmember_binding"]
                        for m in members:
                            if "servername" in m and m["servername"] == name:
                                self.info(
                                    f"Server {name!r} part of Servicegroup {sg_dict['servicegroupname']!r}, skip delete")
                                return False
                except Exception as err:
                    self.warn(
                        f"exception seen while querying sg bindings: {err!s}")
        else:
            self.warn("failed to query servicegroups.. force delete server")

        self.info("Delete server %s", name)
        res2 = self.__nitro.del_server(name)
        if self.__parser.success(res2):
            return True
        return False

    def remove_group(self, group):
        self.info("Delete servicegroup: %s", group.name)
        sg_json = self.__nitro.get_servicegroup(group.name)
        if self.__parser.success(sg_json):
            res = self.__nitro.del_servicegroup(group.name)
            self._raise_on_fail(res)
        else:
            self.warn("servicegroup '%s' doesn't exists.", group.name)

    def get_group(self, group_name):
        self.info("Check if servicegroup %s exists", group_name)
        sg_json = self.__nitro.get_servicegroup(group_name)
        if self.__parser.success(sg_json):
            self.info("servicegroup exists")
            return True
        else:
            self.info("servicegroup does not exist")
            return False

    def get_server_status_in_group(self, server, group):
        out = self.__nitro.get_sg_sgm_b(group.name)
        if self.__parser.success(out) and "servicegroup_servicegroupmember_binding" in out and len(
                out["servicegroup_servicegroupmember_binding"]) > 0:
            members = out["servicegroup_servicegroupmember_binding"]
            mem_filter = [
                m for m in members if str(
                    m['servername']) == str(
                    server.name) and int(
                    m['port']) == int(
                    group.port)]
            if mem_filter:
                return mem_filter[0]['svrstate']
        self.warn(
            "server %s status in group %s not found. nitro_output: %s",
            server.name,
            group.name,
            out)
        raise config_failed("failed to fetch server state in group.")

    def get_ha_node_state(self):
        out = self.__nitro.get_hanode(elem_id="0")
        self._raise_on_fail(out)
        if 'hanode' in out and len(out['hanode']) == 1:
            return out['hanode'][0]['state']
        raise config_failed("failed to fetch state from hanode %s." % out)

    def get_node_config(self):
        CCO_NODE = 0x00000040
        MASTER_NODE = 0x00000004
        out = self.__nitro.get_nsconfig()
        if self.__parser.success(out) and 'nsconfig' in out:
            node_cco = str(out['nsconfig'])
            if node_cco == 'NON-CCO':
                return 'CL Node'
            ns_systemtype = str(out['nsconfig']['systemtype'])
            nc_flags = int(out['nsconfig']['flags'])
            if ns_systemtype == 'Cluster':
                return ('CCO' if (nc_flags & CCO_NODE) == CCO_NODE else 'CL Node')
            elif ns_systemtype == 'HA':
                return ('Primary' if (nc_flags & MASTER_NODE) == MASTER_NODE else 'Secondary')
            return 'StandAlone'
        return 'None'

    def get_servers_in_group(self, group):
        out = self.__nitro.get_sg_sgm_b(group.name)
        result = []
        if self.__parser.success(out) and "servicegroup_servicegroupmember_binding" in out and len(
                out["servicegroup_servicegroupmember_binding"]) > 0:
            members = out["servicegroup_servicegroupmember_binding"]
            members = [m for m in members if m["servername"]]
            for m in members:
                this_server = server()
                this_server.name = m["servername"]
                this_server.ip = m["ip"]
                this_server.state = m["svrstate"]
                result.append(this_server)
                self.debug(
                    "Local group %s contains: %s %s %s",
                    m["servicegroupname"],
                    m["servername"],
                    m["svrstate"],
                    m["ip"])
        else:
            self.debug("Local group %s is empty.", group.name)
        return result

    def add_server_to_group(self, server, group):
        self.info("Add %s %s to %s", server.name, server.ip, group.name)
        self.debug("Check if server %s already exists", server.ip)
        out = self.__nitro.get_server(server.name)
        if self.__parser.no_such_resource(out):
            self.debug("Create new server %s %s", server.name, server.ip)
            out2 = self.__nitro.add_server(server.name, server.ip)
            self._raise_on_fail(out2)

        self.debug("Check if service group %s already exists", group.name)
        sg_json = self.__nitro.get_servicegroup(group.name)
        if self.__parser.no_such_resource(sg_json):
            self.debug("Create new servicegroup %s", group.name)
            res = self.__nitro.add_servicegroup(group.name, group.protocol)
            self._raise_on_fail(res)

        self.debug("Bind %s:%s to %s", server.name, group.port, group.name)
        res = self.__nitro.add_sg_sgm_b(
            sg_name=group.name,
            port=group.port,
            s_name=server.name)
        if self.__parser.already_exist(res):
            self.debug(
                "Bind %s:%s to %s already exists",
                server.name,
                group.port,
                group.name)
        else:
            self._raise_on_fail(res)

    def remove_server_from_group(self, server, group):
        # Do not call unbind in case of cluster setup, BUG: NSPLAT-16894
        nsconfig_info = self.__nitro.get_nsconfig()
        ns_systemtype = str(nsconfig_info['nsconfig']['systemtype'])
        if ns_systemtype != 'Cluster':
            self.info("Unbind %s:%s from %s",
                      server.name, group.port, group.name)
            res = self.__nitro.del_sg_sgm_b(
                sg_name=group.name,
                port=group.port,
                s_name=server.name)
            if not self.__parser.no_such_resource(res):
                self._raise_on_fail(res)

        self.remove_server(server.name)

    def drain_server_in_group(self, server, group, drain_time, graceful="NO"):
        res = self.__nitro.disable_servicegroup(
            sg_name=group.name,
            s_name=server.name,
            port=group.port,
            delay=drain_time,
            graceful=graceful)
        self._raise_on_fail(res)
        self.info("Disabled '%s' in '%s', draining for %d seconds",
                  server.name, group.name, int(drain_time))

    def enable_server_in_group(self, server, group):
        res = self.__nitro.enable_servicegroup(
            sg_name=group.name,
            s_name=server.name,
            port=group.port)
        if self.__parser.success(res):
            self.info("Enabled %s in %s", server.name, group.name)
            return True
        return False

    def get_lbs_in_ns(self):
        try:
            return self.__nitro.get_lbvserver_count()
        except Exception as e:
            self.warn(
                "Failed to get lbserver count configured in NS: '%s'",
                str(e))
        return 0

    def get_sgs_in_ns(self):
        try:
            return self.__nitro.get_servicegroup_count()
        except Exception as e:
            self.warn(
                "Failed to get servicegroup count configured in NS: '%s'",
                str(e))
        return 0

    def get_services_in_ns(self):
        try:
            return self.__nitro.get_service_count()
        except Exception as e:
            self.warn(
                "Failed to get services count configured in NS: '%s'",
                str(e))
        return 0

    def add_lb(self, lb):
        self.info("Add lbverserver: '%s'", lb.name)
        res = self.__nitro.add_lbvserver(lb.name, lb.protocol, lb.ip, lb.port)
        if self.__parser.already_exist(res):
            self.warn("lbvserver '%s' already exists", lb.name)
        else:
            self._raise_on_fail(res)

    def remove_lb(self, lb):
        self.info("Delete lbverserver: '%s'", lb.name)
        res = self.__nitro.del_lbvserver(lb.name)
        if self.__parser.no_such_resource(res):
            self.warn("lbvserver '%s' doesn't exists.", lb.name)
        else:
            self._raise_on_fail(res)

    def get_lb_exist(self, lb_name) -> bool:
        sg_json = self.__nitro.get_lbvserver(lb_name)
        return bool(self.__parser.success(sg_json))

    def add_group_to_lb(self, group, lb):
        self.info("Add NS binding %s to %s", group.name, lb.name)
        lbvs_j = self.__nitro.get_lbvserver(lb.name)
        if self.__parser.no_such_resource(lbvs_j):
            self.warn("lbvserver '%s' doesn't exists. Adding new.", lb.name)
            self.add_lb(lb)
            lbvs_j = self.__nitro.get_lbvserver(lb.name)
        bind_j = self.__nitro.add_lbvs_sgm_b(lb.name, group.name)
        if self.__parser.already_exist(bind_j):
            self.warn(
                "lbvserver '%s', group '%s' binding already exists",
                lb.name,
                group.name)
        else:
            self._raise_on_fail(bind_j)

    def remove_group_from_lb(self, group, lb):
        self.info("Remove NS binding %s to %s", group.name, lb.name)
        bind_j = self.__nitro.del_lbvs_sgm_b(lb.name, group.name)
        if self.__parser.entity_not_bound(bind_j):
            self.warn("lbvserver '%s', servicegroup '%s' binding is missing. Can't delete.",
                      lb.name, group.name)
        else:
            self._raise_on_fail(bind_j)

    def get_lb_sg_b_exist(self, lb_name, sg_name) -> bool:
        bind_j = self.__nitro.get_lbvs_sgm_b(lb_name)
        if self.__parser.success(bind_j) and "lbvserver_servicegroup_binding" in bind_j:
            for b in bind_j["lbvserver_servicegroup_binding"]:
                if b['servicegroupname'] == sg_name:
                    return True
        return False

    def move_vservers_from_sg1_to_sg2(self, sg1, sg2):
        self.info("Moving NS bindings from %s to %s", sg1.name ,sg2.name)
        sg1_binds = self.__nitro.get_sg_b(sg1.name)
        self._raise_on_fail(sg1_binds)
        for sg1_bind in sg1_binds["servicegroupbindings"]:
            if "vservername" in sg1_bind:
                self.info("Moving NS vserver binding %s from %s to %s", sg1_bind["vservername"], sg1.name ,sg2.name)
                res = self.__nitro.del_lbvs_sgm_b(sg1_bind["vservername"], sg1.name)
                if not self.__parser.success(res):
                    self.error("nitro cli del failed with output '%s'", res)
                res = self.__nitro.add_lbvs_sgm_b(sg1_bind["vservername"], sg2.name)
                if not self.__parser.success(res):
                    self.error("nitro cli add failed with output '%s'", res)

    def get_nsrunningconfig(self) -> list:
        out = self.__nitro.get_nsrunningconfig()
        if self.__parser.success(out):
            if "nsrunningconfig" in out and "response" in out['nsrunningconfig']:
                ns_r_c = out['nsrunningconfig']['response'].split("\n")
                while ns_r_c[-1] in ('', ' Done'):
                    ns_r_c.pop()
                return ns_r_c
        return []

    def get_system_stat(self, name):
        return self.__nitro.get_system_stat(name)

    def get_status_of_instance_in_asg(self, instance_name, asg_name):
        out = self.__nitro.get_sg_sgm_b(asg_name)
        if self.__parser.success(out) and "servicegroup_servicegroupmember_binding" in out and len(
                out["servicegroup_servicegroupmember_binding"]) > 0:
            members = out["servicegroup_servicegroupmember_binding"]
            mem_filter = [m for m in members if str(
                m['servername']) == str(instance_name)]
            if mem_filter:
                return mem_filter[0]['svrstate']
            else:
                return 'server_not_found'
        self.warn(
            "server %s status in group %s not found. nitro_output: %s",
            instance_name,
            asg_name,
            out)
        raise config_failed("failed to fetch server state in group.")

    def remove_arp_entry(self, ip):
        self.info("check if arp entry exists for %s", ip)
        res = self.__nitro.get_arp(ip)
        if self.__parser.success(res) and 'arp' in res:
            owner_nodes = len(res['arp'])
            self.info("%d owner nodes present", owner_nodes)
            for i in range(owner_nodes):
                self.info("Delete arp entry for ip: %s, ownernode: %s",
                          ip, str(res['arp'][i]["ownernode"]))
                res = self.__nitro.del_arp(ip, res['arp'][0]["ownernode"])
                if self.__parser.success(res):
                    self.info("deleted ARP")
            return True
        return False
