#!/bin/bash
#
# Copyright 2014 NetScaler, Inc. All rights reserved.
# This software and documentation contain valuable trade
# secrets and proprietary property belonging to Citrix Systems.
# None of this software and documentation may be copied,
# duplicated or disclosed without the express written
# permission of Citrix Systems.
#
# This is the script to generate tech-support bundle for Linux based NSGClient.
# Functions:
#       0. Create tech-support bundle directory
#               0.1 bundle := {nsgclient} ## {_techsupport_} ## {date:yyyymmdd_hh_mm} ## .tgz
#               0.2 Create $(bundle) directory in "~/.citrix" folder
#       1. Fetch all NSGClient relevant information from the System
#               1.1 OS information
#               1.2 System information
#               1.3 Interface configuration(s)
#               1.4 Route information
#       2. Send email to mail list about build status.
#
# Assumptions:
#         1. PATH variable has all pathes required
#         2. All programs required for successfull run are installed
#         3. The user has proper SUDO access

function do_techsupport_bundle()
{
    local answer;

    if [[ $EUID -eq 0 ]]; then
    while true; do
        read -p "Collect tech-support for 'root' user (Y/N)?" answer;
        case $answer in 
            [Yy]*)
                break ;
            ;;
            *)
                exit 100 ;
            ;;
        esac;
    done
    fi

    local usrid=$USER;
    local usrhome=$HOME;
    echo "usrid: $usrid";
    echo "usrhome: $usrhome";

    if [[ ! -d ~/.citrix ]]; then
        echo "ERROR: Directory '~/.citrix' does not exist." 1>&2 ;
        exit 100;
    fi

    cd ~/.citrix ;
    if [[ $? -ne 0 ]]; then
        echo "ERROR: Cannot cd to '~/.citrix'" 1>&2 ;
        exit 100;
    fi

    sleep 1 ;
    local bundle=nsgclient_techsupport_$(date "+%Y%m%d_%H_%M");
    echo "Creating tech-support bundle: ~/.citrix/$bundle";
    mkdir $bundle ;
    cd $bundle ;
    if [[ $? -ne 0 ]]; then
        echo "ERROR: Cannot cd to '~/.citrix/$bundle'" 1>&2 ;
        exit 100;
    fi

    ### USER/SYSTEM CONFIGURATIONS ###
    #~/.citrix/userConfiguration.conf
    #/opt/Citrix/NSGClient/globalConfiguration.conf
    #/opt/Citrix/NSGClient/globalConfiguration.json
    cp ~/.citrix/userConfiguration.conf ./
    sudo cp /opt/Citrix/NSGClient/globalConfiguration.conf ./
    sudo cp /opt/Citrix/NSGClient/globalConfiguration.json ./

    ### SYSTEM/ROUTE/INTERFACE CONFIGURATIONS ###
    #uname -a
    #lsb_release -a
    #route -rn
    #ifconfig -a

    cat /dev/null >./README ;
    if [[ -f $(which uname) ]]; then
        echo "+---------------------------------------------------------------" >>./README ;
        echo "| System Information: "                                           >>./README ;
        echo "+---------------------------------------------------------------" >>./README ;
        uname -a                                                                >>./README ;
    fi

    echo ""                                                                     >>./README ;
    if [[ -f $(which lsb_release) ]]; then
        echo "+---------------------------------------------------------------" >>./README ;
        echo "| Release Distribution Information: "                             >>./README ;
        echo "+---------------------------------------------------------------" >>./README ;
        lsb_release -a                                                          >>./README ;
    fi

    echo ""                                                                     >>./README ;
    if [[ -f $(which route) ]]; then
        echo "+---------------------------------------------------------------" >>./README ;
        echo "| IP routing table: "                                             >>./README ;
        echo "+---------------------------------------------------------------" >>./README ;
        route                                                                   >>./README ;
        echo "+---------------------------------------------------------------" >>./README ;
        ip route                                                                >>./README ;
    fi

    echo ""                                                                     >>./README ;
    if [[ -f $(which ifconfig) ]]; then
        echo "+---------------------------------------------------------------" >>./README ;
        echo "| Interface Configuration: "                                      >>./README ;
        echo "+---------------------------------------------------------------" >>./README ;
        ifconfig -a                                                             >>./README ;
    fi

    ### USER LOGS ###
    #> ls -1 browserlog.* userConfiguration.conf nssslvpn.txt nsepa.txt 
    #~/.citrix/browserlog.agLinuxEpa
    #~/.citrix/browserlog.agLinuxVpnd
    #~/.citrix/nsepa.txt
    #~/.citrix/nssslvpn.txt
    cp ~/.citrix/browserlog.* ./
    cp ~/.citrix/nsepa.txt ./
    cp ~/.citrix/*nssslvpn*.txt ./
    cp ~/.citrix/*core* ./
    cp /var/crash/*NSGClient* ./
    cp /var/log/service* ./
    cp /var/lib/apport/coredump/core._opt_Citrix_NSGClient_bin_NSGClient* ./

    ### KERNEL LOGS ###
    #if (UBUNTU)
    #       /var/log/kern.log
    #       /var/log/syslog
    #fi
    local osflavor=$(lsb_release -a | egrep "Distributor ID" | awk -F: '{ print $NF }' | sed -e 's/^[ \t]\+//g');
    case "$osflavor" in
        Ubuntu)
            sudo cp /var/log/kern.log ./
            sudo cp /var/log/syslog ./
        ;;
    esac;

    ### TODO: PROXY LOGS ###

    ### TODO: NSGClient & Kernel Module core files ###

    usrgrp=$(ls -ld $usrdir | awk '{ print $4 }') ;
    echo "usrid: $usrid - usrgrp: $usrgrp";
    sudo chown $usrid:$usrgrp ./globalConfiguration.conf ;
    sudo chown $usrid:$usrgrp ./globalConfiguration.json ;

    cd ~/.citrix ;
    sudo chown -R $usrid:$usrgrp $bundle
    tar czvf $bundle.tgz $bundle/
    rm -rf $bundle
    sudo chown $usrid:$usrgrp $bundle.tgz
	
    echo "Created tech-support bundle: ~/.citrix/$bundle.tgz";
}

do_techsupport_bundle
